<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;  // For password hashing
use Illuminate\Support\Str; 
use Illuminate\Support\Facades\Log;

class AdminUserController extends Controller
{

// add user
    public function store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'sName' => 'required|string|max:255',
                'sMobileNo' => 'required|string|max:15',
                'sPassword' => 'required|string|min:4',  
                'sEmailId' => 'required|email|max:255',
                'sRole' => 'nullable|string|max:255', 
                'sAddress' => 'nullable|string',
                'iBisActive' => 'nullable|integer',
                'dtUpdatedDate' => 'nullable|date',
                'sOtherDetails1' => 'nullable|string|max:255',
                'sOtherDetails2' => 'nullable|string|max:255',
                'sOtherDetails3' => 'nullable|string|max:255',
                'sOtherDetails4' => 'nullable|string|max:255',
            ]);
        } catch (\Exception $e) {
            Log::error('Validation Error: ' . $e->getMessage());
            return response()->json(['result' => 0, 'message' => 'Validation failed', 'error' => $e->getMessage()], 422);
        }
    
        $existingMobile = DB::table('tblAdminUser')->where('sMobileNo', $validatedData['sMobileNo'])->exists();
        if ($existingMobile) {
            Log::info('Mobile number already registered: ' . $validatedData['sMobileNo']);
            return response()->json([
                'result' =>"6",
                'message' => 'Mobile number already registered',
            ], 409);  
        }
    
        $existingEmail = DB::table('tblAdminUser')->where('sEmailId', $validatedData['sEmailId'])->exists();
        if ($existingEmail) {
            Log::info('Email already registered: ' . $validatedData['sEmailId']);
            return response()->json([
                'result' => "6",
                'message' => 'Email already registered',
            ], 409);  
        }
    
        $salt = Str::random(16); 
    
        $hashedPassword = hash('sha256', $validatedData['sPassword'] . $salt);
    
        $registrationData = [
            'sName' => $validatedData['sName'],
            'sMobileNo' => $validatedData['sMobileNo'],
            'sPassword' => $hashedPassword,
            'Salt' => $salt,  // Store the salt
            'sEmailId' => $validatedData['sEmailId'],
            'sAddress' => $validatedData['sAddress'] ?? null,
            'sRole' => $validatedData['sRole'] ?? null,
            'iBisActive' => $validatedData['iBisActive'] ?? 1,
            'dtUpdatedDate' => now(),
        ];
    
        try {
            $id = DB::table('tblAdminUser')->insertGetId($registrationData);
            Log::info('Registration created successfully with ID: ' . $id);
            return response()->json([
                'result' => "2",
                'message' => 'Registration created successfully',
                'id' => $id,
                'data' => $registrationData,
            ], 201);  
        } catch (\Exception $e) {
            Log::error('Database insert failed: ' . $e->getMessage());
            return response()->json([
                'result' => "5",
                'message' => 'Error occurred while creating registration',
                'error' => $e->getMessage(),
            ], 500); 
        }
    }
    


    // Update User
    public function update(Request $request)
    {
        // Validate the incoming request data
        $validatedData = $request->validate([
            'sName' => 'nullable|string|max:255',
            'sMobileNo' => 'nullable|string|max:15',
            'sEmailId' => 'nullable|email|max:255',
            'sRole' => 'nullable|string|max:255',
            'sAddress' => 'nullable|string',
            'iBisActive' => 'nullable|integer',
            'dtUpdatedDate' => $request->dtUpdatedDate ?? now(),
        ]);
        $iId =  $request->input('iId');

        // Fetch the user by ID
        $user = DB::table('tblAdminUser')->where('iId', $iId)->first();

        // Prepare the data for update
        $updateData = [
            'sName' => $request->sName ?? $user->sName,
            'sMobileNo' => $request->sMobileNo ?? $user->sMobileNo,
            'sEmailId' => $request->sEmailId ?? $user->sEmailId,
            'sAddress' => $request->sAddress ?? $user->sAddress,
            'sRole' => $request->sRole ?? $user->sRole,
            'iBisActive' => $request->iBisActive ?? $user->iBisActive,
            'dtUpdatedDate' => $request->dtUpdatedDate ?? now(),  // Current timestamp if not provided
        ];

        // Update the data in the database
        try {
            DB::table('tblAdminUser')->where('iId', $iId)->update($updateData);

            return response()->json([
                'result' => "3",
                'message' => 'User updated successfully',
                'data' => $updateData,
            ], 200);  // 200 OK
        } catch (\Exception $e) {
            return response()->json([
                'result' =>"5",
                'message' => 'Error occurred while updating user',
                'error' => $e->getMessage(),
            ], 500);  // 500 Internal Server Error
        }
    }




    // Delete Admin User
    public function destroy($id)
    {
        // Find the user by ID
        $user = DB::table('tblAdminUser')->where('iId', $id)->first();

        if (!$user) {
            return response()->json(['result' => 0, 'message' => 'User not found'], 404);
        }

        $updated = DB::table('tblAdminUser')->where('iId', $id)->update(['iBisActive' => 0]);

        if ($updated) {
            return response()->json(['result' => "4", 'message' => 'User marked as inactive successfully'], 200);
        } else {
            return response()->json(['result' => "5", 'message' => 'Failed to mark user as inactive'], 500);
        }
    }




    // Get active admin users
    public function index()
    {
        // Fetch users where iBisActive = 1
        $activeUsers = DB::table('tblAdminUser')->where('iBisActive', 1)->get();

        if ($activeUsers->isEmpty()) {
            return response()->json([
                'result' => 0,
                'message' => 'No active users found',
            ], 404);  // 404 Not Found if no active users are found
        }

        return response()->json([
            'result' => "1",
            'message' => 'Active users retrieved successfully',
            'data' => $activeUsers,
        ], 200);  // 200 OK
    }



    // Login
    public function loginAdminUser(Request $request)
    {
        // Prepare the output array
        $output = [];

        try {
            // Validate the incoming request data
            $validatedData = $request->validate([
                'sMobileNo' => 'required|string|max:15',
                'sPassword' => 'required|string|min:4',
            ]);

            // Get mobile number and password from the validated data
            $sMobileNo = $validatedData['sMobileNo'];
            $sPassword = $validatedData['sPassword'];

            // Retrieve user from tblAdminUser by mobile number
            $user = DB::table('tblAdminUser')->where('sMobileNo', $sMobileNo)->first();

            if ($user) {
                // Hash the input password with the stored salt
                $hashedPassword = hash('sha256', $sPassword . $user->Salt);

                // Check if the hashed password matches the stored password for this user
                if ($hashedPassword === $user->sPassword && $user->iBisActive) {
                    // Success: Valid user and password
                    $output['result'] = "8";
                    $output['records'] = $user;
                } else {
                    // Incorrect password or inactive user
                    $output['result'] = "5";
                    $output['message'] = "Invalid password or user is inactive.";
                }
            } else {
                // If the user does not exist
                $output['result'] = "5";  // User not found
                $output['message'] = "User not found.";
            }

        } catch (\Exception $e) {
            // If any exception occurs, return an error message with exception details
            $output['result'] = "0";  // General error
            $output['message'] = "An error occurred while processing your request.";
            $output['error'] = $e->getMessage();  // You can log this if needed
        }

        // Return the response as JSON
        return response()->json($output);
    }



    // Change Password
    public function changePassword(Request $request)
    {
        // Validate incoming request
        $validatedData = $request->validate([
            'iId' => 'required|integer|exists:tblAdminUser,iId',
            'sPassword' => 'required|string|min:4',
        ]);

        try {
            // Generate a new random salt
            $salt = Str::random(16);

            // Hash the new password with the new salt
            $hashedPassword = hash('sha256', $validatedData['sPassword'] . $salt);

            // Update the password in the database for the given iId
            $updated = DB::table('tblAdminUser')
                ->where('iId', $validatedData['iId'])
                ->update([
                    'sPassword' => $hashedPassword,
                    'Salt' => $salt,  // Update the salt
                    'dtUpdatedDate' => now(),
                ]);

            if ($updated) {
                return response()->json([
                    'result' => "9",
                    'message' => 'Password updated successfully',
                ], 200);
            } else {
                return response()->json([
                    'result' => "5",
                    'message' => 'No changes made or user not found',
                ], 400);
            }
        } catch (\Exception $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Error updating password',
                'error' => $e->getMessage(),
            ], 500);
        }
    }







}