<?php


namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller; 
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class InstituteController extends Controller
{



    // Insert a new record
    public function store(Request $request)
    {
        $data = $request->only([
            'sProgram', 'sInstituteUniqueID', 'iCollegeCode', 'sCourseName', 'sCollegeName', 
            'sAddress', 'sStatus','sState','sDistrict', 'sEstablishmentYear', 'sUniversityName', 
            'sAutonomyStatus', 'sGendorEducation', 'sMinorityStatus', 'semailID', 'sWebsite', 
            'sIntake', 'sPhoneNo','sChoiceCode'	,'sBranchName','sAccreditation', 'iTuitionFeeSem1', 'iDevelopmentFeeSem1','iTotalFeeSem1', 'iTuitionFeeSem2', 'iDevelopmentFeeSem2', 'iTotalFeeSem2', 'sPdfFeedetailsLinkSem1', 
            'sPdfFeedetailsLinkSem2', 'iBisActive', 'dtUpdatedDate', 'sOtherDetails1', 
            'sOtherDetails2', 'sOtherDetails3'
        ]);
        $data['dtUpdatedDate'] = now();
        $data['iBisActive']=1;
        $id = DB::table('tblinstitutedetails')->insertGetId($data);

        return response()->json(['result'=>'2','message' => 'Institute created', 'id' => $id], 201);
    }




// To add Multiple instituts

    public function addInstitutes(Request $request)
    {
        try {
            $institutesData = $request->input('colleges', []); 
            $status = $request->input('status', 'append');
            $sProgram = $request->input('program', []); 
            
            if (empty($institutesData) || !is_array($institutesData)) {
                return response()->json(['result' => '0', 'message' => 'Invalid institute data provided.'], 400);
            }
    
            if ($status === 'addall') {
                if (!empty($sProgram) && is_array($sProgram)) {
                    foreach ($sProgram as $program) {
                        DB::table('tblinstitutedetails')->where('sProgram', $program)->delete(); 
                    }
                } else {
                    DB::table('tblinstitutedetails')->truncate(); 
                }
            }
    
            $currentDate = now();
    
            // Batch size calculation
            $batchSize = 1800;
            $chunks = array_chunk($institutesData, $batchSize);
    
            foreach ($chunks as $chunk) {
                foreach ($chunk as &$data) {
                    $data['dtUpdatedDate'] = $currentDate;
                    $data['iBisActive'] = 1; // Mark as active
                }
    
                DB::table('tblinstitutedetails')->insert($chunk);
            }
    
            return response()->json(['result' => '2', 'message' => 'Institutes added successfully.'], 201);
    
        } catch (\Exception $e) {
            \Log::error('Error in addInstitutes:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json(['result' => '0', 'message' => 'An error occurred while adding institutes.'], 500);
        }
    }
    



    // Update an existing record
    public function update(Request $request)
    {
        $data = $request->only([
            'sProgram', 'sInstituteUniqueID', 'iCollegeCode', 'sCourseName', 'sCollegeName', 
            'sAddress', 'sStatus','sState', 'sDistrict', 'sEstablishmentYear', 'sUniversityName', 
            'sAutonomyStatus', 'sGendorEducation', 'sMinorityStatus', 'semailID', 'sWebsite', 
            'sIntake', 'sPhoneNo','sChoiceCode'	,'sBranchName'	,'sAccreditation', 'iTuitionFeeSem1', 'iDevelopmentFeeSem1', 'iTotalFeeSem1', 
            'iTuitionFeeSem2', 'iDevelopmentFeeSem2', 'iTotalFeeSem2', 'sPdfFeedetailsLinkSem1', 
            'sPdfFeedetailsLinkSem2', 'iBisActive', 'dtUpdatedDate', 'sOtherDetails1', 
            'sOtherDetails2', 'sOtherDetails3'
        ]);
        $iId =  $request->input('iId');
        $data['dtUpdatedDate'] = now();

        $updated = DB::table('tblinstitutedetails')->where('iId', $iId)->update($data);

        if ($updated) {
            return response()->json(['result'=>'3','message' => 'Institute updated']);
        } else {
            return response()->json(['result'=>'5' ,'message' => 'Institute not found'], 404);
        }
    }






    // Delete a record
    public function destroy($iId)
    {
        $deleted = DB::table('tblinstitutedetails')->where('iId', $iId)->update(['iBisActive' => 0]);
        if ($deleted) {
            return response()->json(['result'=>'4','message' => 'Institute deleted']);
        } else {
            return response()->json(['result'=>'5' ,'message' => 'Institute not found'], 404);
        }
    }





     // Get all records
    public function index()
    {
         return response()->json(['result'=>'1','data'=>DB::table('tblinstitutedetails')->where('iBisActive', 1)->get()]);
    }
 
 


    // Get a specific record by ID
    public function show($iId)
    {
        $institute = DB::table('tblinstitutedetails')
        ->where([
            ['iId', '=', $iId],          
            ['iBisActive', '=', 1],    
        ])
        ->first();
            
        if ($institute) {
            return response()->json([
                'result' => '1', 
                'message' => 'Institute found successfully',
                'data' => $institute, 
            ]);
        } else {
            return response()->json([
                'result' => '5',
                'message' => 'Institute not found',
            ], 404);
        }
    }
    




// get collge list by pagination
    public function getPaginatedColleges(Request $request)
    {
        $perPage = $request->query('per_page', 10); 
        $page = $request->query('page', 1); 

        $colleges = DB::table('tblinstitutedetails')
            ->select('*') 
            ->where('iBisActive', 1)
            ->paginate($perPage);

        return response()->json($colleges);
    }



    
// Search OPtion list
    public function search(Request $request)
    {
        $query = $request->input('query');
        $perPage = $request->input('per_page', 10); 

        if (!$query) {
            return response()->json(['error' => 'Query parameter is required'], 400);
        }

        $searchableColumns = [
            'sProgram', 'sInstituteUniqueID', 'iCollegeCode', 'sCourseName', 'sCollegeName',
            'sAddress', 'sStatus', 'sDistrict', 'sEstablishmentYear', 'sUniversityName',
            'sAutonomyStatus', 'sGendorEducation', 'sMinorityStatus', 'semailID', 'sWebsite',
            'sIntake', 'sPhoneNo', 'sChoiceCode'	,'sBranchName'	,'sAccreditation', 'iTuitionFeeSem1', 'iDevelopmentFeeSem1', 'iTotalFeeSem1','iTuitionFeeSem2', 'iDevelopmentFeeSem2', 'iTotalFeeSem2', 'sPdfFeedetailsLinkSem1','sPdfFeedetailsLinkSem2', 'iBisActive', 'dtUpdatedDate', 'sOtherDetails1',
            'sOtherDetails2', 'sOtherDetails3'
        ];

        $results = DB::table('tblinstitutedetails')
        ->where('iBisActive', 1)
            ->where(function ($q) use ($query, $searchableColumns) {
                foreach ($searchableColumns as $column) {
                    $q->orWhere($column, 'LIKE', '%' . $query . '%');
                }
            })
            ->paginate($perPage);

        return response()->json($results);
    }





    // get colleges by sorting
    public function getColleges(Request $request)
    {
        $allowedSortFields = [
            'sCourseName',
            'sCollegeName',
            'sStatus',
            'sDistrict',
            'sEstablishmentYear',
            'sUniversityName',
            'iTotalFeeSem1',
            'iTotalFeeSem2',
            'sIntake',
            'sMinorityStatus'
        ];
    
        $sortField = $request->query('sortField', 'sCollegeName'); // Default sort field
        $sortOrder = $request->query('sortOrder', 'asc'); // Default sort order
    
        if (!in_array($sortField, $allowedSortFields)) {
            return response()->json([
                'error' => 'Invalid sorting field.',
                'allowedSortFields' => $allowedSortFields,
            ], 400);
        }
    
        $perPage = $request->query('per_page', 10);
        $page = $request->query('page', 1);
    
        $colleges = DB::table('tblinstitutedetails')
            ->where('iBisActive', 1)
            ->orderBy($sortField, $sortOrder) 
            ->paginate($perPage, ['*'], 'page', $page);
    
        return response()->json($colleges);
    }
    


    //get College by filters
    public function getCollegesByFilters(Request $request)
    {
        try {
            // Validate request
            $validatedData = $request->validate([
                'program' => 'array|nullable',
                'program.*' => 'string',
                'state' => 'array|nullable',
                'state.*' => 'string',
                'city' => 'array|nullable',
                'city.*' => 'string',
                'status' => 'array|nullable',
                'status.*' => 'string',
                'course' => 'array|nullable',
                'course.*' => 'string',
                'per_page' => 'integer|nullable', 
                'current_page' => 'integer|nullable',
            ]);
    
            // Extract filters and pagination
            $programs = $validatedData['program'] ?? [];
            $states = $validatedData['state'] ?? [];
            $cities = $validatedData['city'] ?? [];
            $statuses = $validatedData['status'] ?? [];
            $courses = $validatedData['course'] ?? [];
            $perPage = $validatedData['per_page'] ?? 10;
            $currentPage = $validatedData['current_page'] ?? 1;
            $offset = ($currentPage - 1) * $perPage;
    
            // Start query
            $query = DB::table('tblinstitutedetails')
                ->selectRaw('
                    iCollegeCode, 
                    MAX(sInstituteName) as sInstituteName, 
                    MAX(sState) as sState, 
                    MAX(sDistrict) as sDistrict, 
                    MAX(sStatus) as sStatus, 
                    MAX(sProgram) as sProgram, 
                    GROUP_CONCAT(DISTINCT sCourseName ORDER BY sCourseName ASC) as sCourses
                ')
                ->where('iBisActive', 1)
                ->groupBy('iCollegeCode'); // Ensure distinct colleges
    
            // Apply filters
            if (!empty($programs)) {
                $query->whereIn('sProgram', $programs);
            }
            if (!empty($states)) {
                $query->whereIn('sState', $states);
            }
            if (!empty($cities)) {
                $query->whereIn('sDistrict', $cities);
            }
            if (!empty($statuses)) {
                $query->whereIn('sStatus', $statuses);
            }
            if (!empty($courses)) {
                $query->whereIn('sCourseName', $courses);
            }
    
            // Get total distinct college count **before pagination**
            $total = DB::table(DB::raw("({$query->toSql()}) as subquery"))
                ->mergeBindings($query)
                ->count();
    
            // Apply pagination
            $colleges = $query->offset($offset)->limit($perPage)->get();
    
            // Calculate last page
            $lastPage = (int) ceil($total / $perPage);
    
            // Return response
            return response()->json([
                'success' => true,
                'data' => $colleges,
                'current_page' => $currentPage,
                'last_page' => $lastPage,
                'total' => $total,
                'per_page' => $perPage,
                'message' => 'Colleges retrieved successfully.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getCollegesByFilters:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving colleges.',
            ], 500);
        }
    }


    // get distinct States with College count
    public function getStates(Request $request)
    {
        // Validate request
        $validatedData = $request->validate([
            'sProgram' => 'array|required',
            'sProgram.*' => 'string'
        ]);
    
        $programs = $request->input('sProgram', []);
    
        $results = DB::table('tblinstitutedetails')
            ->select('sState', DB::raw('COUNT(DISTINCT iCollegeCode) as count'))
            ->where('iBisActive', 1)
            ->whereNotNull('sState')
            ->whereIn('sProgram', $programs)
            ->groupBy('sState')
            ->get();
    
        return response()->json([
            'success' => true,
            'data' => $results,
            'message' => 'States fetched successfully based on Program.'
        ]);
    }



    // get distinct district
    public function getCities()
    {
       $results = DB::table('tblinstitutedetails')
       ->select(DB::raw('DISTINCT sDistrict'), DB::raw('COUNT(sDistrict) as count'))
       ->where('iBisActive', 1)
       ->groupBy('sDistrict')
       ->get();
       
        return response()->json($results);
    }



    // get district by selected states
    public function getDistrictsByFilter(Request $request)
    {
        $request->validate([
            'states'  => 'nullable|array',
            'states.*' => 'string',
            'program' => 'nullable|array',
            'program.*' => 'string',
        ]);
    
        $states = $request->input('states', []);
        $programs = $request->input('program', []);
    
        $query = DB::table('tblinstitutedetails')
            ->select('sDistrict', DB::raw('COUNT(DISTINCT iCollegeCode) as count'))
            ->where('iBisActive', 1)
            ->whereNotNull('sDistrict')
            ->groupBy('sDistrict');
    
        // Apply state filter if provided
        if (!empty($states)) {
            $query->whereIn('sState', $states);
        }
    
        // Apply program filter if provided
        if (!empty($programs)) {
            $query->whereIn('sProgram', $programs);
        }
    
        $result = $query->get();
    
        return response()->json([
            'success' => true,
            'data' => $result,
            'message' => 'Districts fetched successfully based on filters.'
        ]);
    }


    


    // get distinct programs
   public function getProgram()
   {
      $results = DB::table('tblinstitutedetails')
      ->select(DB::raw('DISTINCT sProgram'), DB::raw('COUNT(sProgram) as count'))
      ->whereNotNull('sProgram')
      ->where('sProgram', '!=', '')
      ->groupBy('sProgram')
      ->get();
      
       return response()->json($results);
   }




   // get programs by selected cities
  public function getProgramsByFilter(Request $request)
  {
    \Log::info('Incoming Request:', $request->all());

    $request->validate([
        'sDistrict.selectedCities' => 'nullable|array',
        'sDistrict.selectedCities.*' => 'string',
        'sDistrict.selectedStates' => 'nullable|array',
        'sDistrict.selectedStates.*' => 'string',
    ]);

    $cities = $request->input('sDistrict.selectedCities', []); 
    $states = $request->input('sDistrict.selectedStates', []); 

    $query = DB::table('tblinstitutedetails')
        ->select('sProgram', DB::raw('COUNT(DISTINCT iCollegeCode) as count'))
        ->whereNotNull('sProgram')
        ->where('sProgram', '!=', '')
        ->where('iBisActive', 1)
        ->groupBy('sProgram');

    if (!empty($cities)) {
        \Log::info('Applying City Filter:', $cities);
        $query->whereIn('sDistrict', $cities);
    }

    if (!empty($states)) {
        $query->whereIn('sState', $states);
    }

    $result = $query->get();

    return response()->json([
        'success' => true,
        'data' => $result,
        'message' => 'Programs retrieved successfully.',
    ]);
}

   
   

   // get distinct Courses
   public function getCourse()
   {
      $results = DB::table('tblinstitutedetails')
      ->select(DB::raw('DISTINCT sCourseName'), DB::raw('COUNT(sCourseName) as count'))
      ->groupBy('sCourseName')
      ->get();
      
       return response()->json($results);
   }
    

    //get course by filter 
    public function getCoursesByfilter(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'sDistrict.cities' => 'array|nullable',
                'sDistrict.cities.*' => 'string',
                'sDistrict.states' => 'array|nullable',
                'sDistrict.states.*' => 'string',
                'sDistrict.programs' => 'array|nullable',
                'sDistrict.programs.*' => 'string',
            ]);

            $cities = $request->input('sDistrict.cities', []);
            $states = $request->input('sDistrict.states', []);
            $programs = $request->input('sDistrict.programs', []);

            $query = DB::table('tblinstitutedetails')
                ->select('sCourseName',DB::raw('COUNT(DISTINCT iCollegeCode) as count'))
                ->where('sCourseName', '!=', '')
                ->whereNotNull('sCourseName')
                ->groupBy('sCourseName');

        
            if (!empty($states)) {
                $query->whereIn('sState', $states);
            } 
            if (!empty($programs)) {
                $query->whereIn('sProgram', $programs);
            } 
            if (!empty($cities)) {
                $query->whereIn('sDistrict', $cities);
            }

            $result = $query->get();

            return response()->json($result);

        } catch (\Exception $e) {
            \Log::error('Error in getCoursesByCityAndProgram: ', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching courses.',
            ], 500);
        }
    }




    // get courses by programs
    public function getCoursesByProgram(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'sProgram' => 'array|nullable',
                'sProgram.*' => 'string',
            ]);

            $programs = $request->input('sProgram', []); 

            // Start the query
            $query = DB::table('tblinstitutedetails')
                ->select('sCourseName', DB::raw('COUNT(*) as count'))  
                ->where('sCourseName', '!=', '') 
                ->whereNotNull('sCourseName');   

            // If programs are provided, filter the courses based on programs
            if (!empty($programs)) {
                $query->whereIn('sProgram', $programs);  // Filter by program if available
            }

            // Group the courses by their names
            $query->groupBy('sCourseName');

            // Fetch results
            $result = $query->get();

            // Return response
            return response()->json($result);

        } catch (\Exception $e) {
            \Log::error('Error in getCoursesByProgram: ', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching courses.',
            ], 500);
        }
    }



    // get distinct Branch Name
    public function getBranchName()
    {
        $results = DB::table('tblinstitutedetails')
        ->select(DB::raw('DISTINCT sBranchName'))
        ->groupBy('sBranchName')
        ->get();
        
            return response()->json($results);
    }



    // get distinct branches by course and program
    public function getBranchByCourseAndProgram(Request $request)
    {
        try {
            // Validate the input
            $validatedData = $request->validate([
                'sProgram' => 'array|nullable',
                'sProgram.*' => 'string',
                'sCourseName' => 'array|nullable',
                'sCourseName.*' => 'string',
            ]);

            // Extract inputs
            $programs = $validatedData['sProgram'] ?? [];
            $course = $validatedData['sCourseName'] ?? [];

            // Start the query
            $query = DB::table('tblinstitutedetails')
                ->select('sBranchName', DB::raw('COUNT(*) as count'))
                ->where('sBranchName', '!=', '')
                ->whereNotNull('sBranchName');

            if (!empty($programs)) {
                $query->whereIn('sProgram', $programs);
            }

            if (!empty($course)) {
                $query->whereIn('sCourseName', $course); // Correct column
            }

            // Group the courses by their names
            $query->groupBy('sBranchName');

            // Fetch results
            $result = $query->get();

            // Handle empty results
            if ($result->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'message' => 'No branches found for the given filters.',
                    'data' => [],
                ], 200);
            }

            // Return response
            return response()->json([
                'success' => true,
                'message' => 'Branches fetched successfully.',
                'data' => $result,
            ]);

        } catch (\Exception $e) {
            // Log error with request data
            \Log::error('Error in getBranchByCourseAndProgram:', [
                'request' => $request->all(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching courses.',
            ], 500);
        }
    }




    // get branch by all filter
   public function getBranchByFilter(Request $request)
    {
    try {
        $validatedData = $request->validate([
            'sProgram' => 'array|nullable',
            'sProgram.*' => 'string',
            'sDistrict' => 'array|nullable',
            'sDistrict.*' => 'string',
            'sState' => 'array|nullable',
            'sState.*' => 'string',
            'sCourseName' => 'array|nullable',
            'sCourseName.*' => 'string',
        ]);

        $cities = $request->input('sDistrict', []); 
        $states = $request->input('sState', []);    
        $programs = $request->input('sProgram', []); 
        $courses = $request->input('sCourseName', []); 

        $query = DB::table('tblinstitutedetails')
            ->select('sBranchName', DB::raw('COUNT(DISTINCT iCollegeCode) as count'))
            ->where('iBisActive', 1)
            ->whereNotNull('sBranchName')
            ->where('sBranchName', '!=', '');

        if (!empty($cities)) {
            $query->whereIn('sDistrict', $cities);
        }

        if (!empty($states)) {
            $query->whereIn('sState', $states);
        }

        if (!empty($programs)) {
            $query->whereIn('sProgram', $programs);
        }

        if (!empty($courses)) {
            $query->whereIn('sCourseName', $courses);
        }

        $result = $query->groupBy('sBranchName')->get();

        return response()->json([
            'success' => true,
            'message' => $result->isEmpty() ? 'No branches found for the given filters.' : 'Branches fetched successfully.',
            'data' => $result,
        ], 200);

    } catch (\Exception $e) {
        \Log::error('Error in getBranchByFilter:', [
            'request' => $request->all(),
            'error' => $e->getMessage(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while fetching branches.',
        ], 500);
    }
}



   // get distinct Courses
   public function getStatus()
   {
      $results = DB::table('tblinstitutedetails')
      ->select(DB::raw('DISTINCT sStatus'), DB::raw('COUNT(sStatus) as count'))
      ->groupBy('sStatus')
      ->get();
      
       return response()->json($results);
   }
    
   
   

    // get stsatus by filter
    public function getStatusByfilter(Request $request)
    {
        // Validate the incoming request
        $validatedData = $request->validate([
            'sProgram' => 'array|nullable',
            'sProgram.*' => 'string',
            'sDistrict' => 'array|nullable',
            'sDistrict.*' => 'string',
            'sState' => 'array|nullable',
            'sState.*' => 'string',
            'sCourseName' => 'array|nullable',
            'sCourseName.*' => 'string',
            'sBranchName' => 'array|nullable',
            'sBranchName.*' => 'string'
        ]);

        // Retrieve filters from the request
        $cities = $request->input('sDistrict', []);  
        $states = $request->input('sState', []);    
        $programs = $request->input('sProgram', []); 
        $courses = $request->input('sCourseName', []);
        $branch = $request->input('sBranchName', []);

        // Build the query dynamically
        $query = DB::table('tblinstitutedetails')
            ->select('sStatus', DB::raw('COUNT(DISTINCT iCollegeCode) as count'))
            ->where('iBisActive', 1)
            ->where('sStatus', '!=', '')
            ->whereNotNull('sStatus') 
            ->groupBy('sStatus');

        // Apply filters dynamically
        if (!empty($cities)) {
            $query->whereIn('sDistrict', $cities);
        }
        if (!empty($states)) {
            $query->whereIn('sState', $states);
        }
        if (!empty($programs)) {
            $query->whereIn('sProgram', $programs);
        }
        if (!empty($courses)) {
            $query->whereIn('sCourseName', $courses);
        }
        if (!empty($branch)) {
            $query->whereIn('sBranchName', $branch);
        }

        // Execute the query and get the results
        $result = $query->get();

        // Return the results as a JSON response with 'data' key
        return response()->json([
            'success' => true,
            'data' => $result,
            'message' => 'Status retrieved successfully.'
        ]);
    }



 

    // get coleges by sort search and filter
    public function getCollegesByFilter(Request $request)
    {
        try {
            // Validate the incoming request dynamically
            $validatedData = $request->validate([
                'query' => 'nullable|string',
                'per_page' => 'integer|nullable', 
                'current_page' => 'integer|nullable', 
                'sortField' => 'nullable|string', 
                'sortOrder' => 'nullable|string', 
                'filters' => 'nullable|array', 
                'filters.state' => 'nullable|array',
                'filters.city' => 'nullable|array',
                'filters.program' => 'nullable|array',
                'filters.course' => 'nullable|array',
                'filters.branch' => 'nullable|array',
                'filters.status' => 'nullable|array',
            ]);
    
            // Extract parameters from the request
            $query = $validatedData['query'] ?? null;
            $perPage = $validatedData['per_page'] ?? 10;
            $currentPage = $validatedData['current_page'] ?? 1;
            $sortField = $validatedData['sortField'] ?? null;
            $sortOrder = $validatedData['sortOrder'] ?? null;
            $filters = $validatedData['filters'] ?? [];
            $offset = ($currentPage - 1) * $perPage;
    
            // Build query to fetch distinct colleges
            $queryBuilder = DB::table('tblinstitutedetails')
                ->selectRaw('
                    iCollegeCode,
                    MIN(iId) as iId,
                    MAX(sInstituteUniqueID) as sInstituteUniqueID,
                    MAX(sCollegeName) as sCollegeName,
                    MAX(sAddress) as sAddress,
                    MAX(sStatus) as sStatus,
                    MAX(sState) as sState,
                    MAX(sDistrict) as sDistrict,
                    MAX(sEstablishmentYear) as sEstablishmentYear,
                    MAX(sUniversityName) as sUniversityName,
                    MAX(sAutonomyStatus) as sAutonomyStatus,
                    MAX(sGendorEducation) as sGendorEducation,
                    MAX(sMinorityStatus) as sMinorityStatus,
                    MAX(semailID) as semailID,
                    MAX(sWebsite) as sWebsite,
                    MAX(sIntake) as sIntake,
                    MAX(sPhoneNo) as sPhoneNo,
                    MAX(sChoiceCode) as sChoiceCode,
                    MAX(sBranchName) as sBranchName,
                    MAX(sAccreditation) as sAccreditation,
                    MAX(iTuitionFeeSem1) as iTuitionFeeSem1,
                    MAX(iDevelopmentFeeSem1) as iDevelopmentFeeSem1,
                    MAX(iTotalFeeSem1) as iTotalFeeSem1,
                    MAX(iTuitionFeeSem2) as iTuitionFeeSem2,
                    MAX(iDevelopmentFeeSem2) as iDevelopmentFeeSem2,
                    MAX(iTotalFeeSem2) as iTotalFeeSem2,
                    GROUP_CONCAT(DISTINCT sCourseName ORDER BY sCourseName ASC) as sCourses,
                    MAX(sProgram) as sProgram
                ')
                ->where('iBisActive', 1)
                ->groupBy('iCollegeCode'); // Ensuring distinct colleges
    
            // Apply filters if provided
            if (!empty($query)) {
                $queryBuilder->where(function ($q) use ($query) {
                    $searchableColumns = [
                        'sProgram', 'sInstituteUniqueID', 'iCollegeCode', 'sCourseName', 'sCollegeName',
                        'sAddress', 'sStatus', 'sState', 'sDistrict', 'sEstablishmentYear', 'sUniversityName',
                        'sAutonomyStatus', 'sGendorEducation', 'sMinorityStatus', 'semailID', 'sWebsite', 'sBranchName', 'sAccreditation'
                    ];
                    foreach ($searchableColumns as $column) {
                        $q->orWhere($column, 'LIKE', '%' . $query . '%');
                    }
                });
            }
    
            if (!empty($filters['state'])) {
                $queryBuilder->whereIn('sState', $filters['state']);
            }
            if (!empty($filters['city'])) {
                $queryBuilder->whereIn('sDistrict', $filters['city']);
            }
            if (!empty($filters['program'])) {
                $queryBuilder->whereIn('sProgram', $filters['program']);
            }
            if (!empty($filters['course'])) {
                $queryBuilder->whereIn('sCourseName', $filters['course']);
            }
            if (!empty($filters['branch'])) {
                $queryBuilder->whereIn('sBranchName', $filters['branch']);
            }
            if (!empty($filters['status'])) {
                $queryBuilder->whereIn('sStatus', $filters['status']);
            }
    
            // Apply sorting if provided
            if ($sortField && $sortOrder) {
                $queryBuilder->orderBy($sortField, $sortOrder);
            }
    
            // Get total count of distinct colleges (using a subquery)
            $total = DB::table(DB::raw("({$queryBuilder->toSql()}) as subquery"))
                ->mergeBindings($queryBuilder)
                ->count();
    
            // Apply pagination
            $colleges = $queryBuilder->offset($offset)->limit($perPage)->get();
    
            // Calculate last page
            $lastPage = (int) ceil($total / $perPage);
    
            // Return response
            return response()->json([
                'success' => true,
                'data' => $colleges,
                'current_page' => $currentPage,
                'last_page' => $lastPage,
                'total' => $total,
                'per_page' => $perPage,
                'message' => 'Colleges retrieved successfully.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getCollegesByFilters:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving colleges.',
            ], 500);
        }
    }



    // get distinct branch ny iCollegeCode
public function getDistinctBranchByCode(Request $request)
{
    try {
        // Validate the input
        $validatedData = $request->validate([
            'iCollegeCode' => 'required|string',
        ]);

        // Extract inputs
        $collegecode = $validatedData['iCollegeCode'];

        // Query to fetch distinct branch names and their corresponding intake
        $result = DB::table('tblinstitutedetails')
            ->select('sBranchName', DB::raw('MAX(sIntake) as sIntake')) // Ensure correct intake selection
            ->where('iCollegeCode', $collegecode)
            ->groupBy('sBranchName') // Group by branch to ensure distinct branch names
            ->get();

        // Handle empty results
        if ($result->isEmpty()) {
            return response()->json([
                'success' => true,
                'message' => 'No branches found.',
                'data' => [],
            ], 200);
        }

        // Return response
        return response()->json([
            'success' => true,
            'message' => 'Branches fetched successfully.',
            'data' => $result,
        ]);

    } catch (\Exception $e) {
        // Log error with request data
        \Log::error('Error in getDistinctBranchByCode:', [
            'request' => $request->all(),
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while fetching branches.',
        ], 500);
    }
}

    
    

    





}
