<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class cutOffController extends Controller
{

    //get all cutOffList
    public function getAllCutOffList()
    {
        $cutofflist=DB::table('tblCutOff')->leftjoin('tblinstitutedetails','tblCutOff.iCollegeCode', '=','tblinstitutedetails.iCollegeCode')
        ->select('tblCutOff.*','tblinstitutedetails.sCollegeName')
        ->where('tblCutOff.iBisActive', 1)
        ->get();
        return response()->json(['result'=>'1','data'=>$cutofflist]);
    }


    // get cutoff list by id
    public function getCutOffByid($iId)
    {
        $cutoff = DB::table("tblCutOff")
        ->leftjoin('tblinstitutedetails','tblCutOff.iCollegeCode', '=','tblinstitutedetails.iCollegeCode')
        ->select('tblCutOff.*','tblinstitutedetails.sCollegeName')
        ->where('tblCutOff.iId',$iId)->first();


        if($cutoff){
            return response()->json([
                'result'=>"1",
                'message'=>'Cutoff found',
                'data'=> $cutoff
            ]);
        }
        else{
                return response()->json([
                    'result'=>"5",
                    'message'=>'Cutoff not found'
                ],404);
            }
    }



    // Add Cut off of Institute
    public function addMultipleCutOff(Request $request)
    {
        try {
            $institutesData = $request->input('cutofflist', []); 
            $status = $request->input('status'); 
            $year = $request->input('iYear');    
            $program = $request->input('sProgram');
    
            if (empty($institutesData) || !is_array($institutesData)) {
                return response()->json(['result' => '0', 'message' => 'Invalid institute data provided.'], 400);
            }
    
            if (empty($status)) {
                return response()->json(['result' => '0', 'message' => 'Status is required.'], 400);
            }
    
            if (empty($year)) {
                return response()->json(['result' => '0', 'message' => 'Year is required.'], 400);
            }
    
            $currentDate = now();
    
            // Handle `status` logic
            if ($status === 'addall') {
                if (!empty($program)) {
                    DB::table('tblCutOff')
                        ->where('sProgram', $program)
                        ->where('iYear', $year)
                        ->delete();
                } else {
                    DB::table('tblCutOff')
                        ->where('iYear', $year)
                        ->delete();
                }
            }
    
            // Batch insert (optimized)
            $batchSize = 1800;  // Adjust batch size if needed
            $chunks = array_chunk($institutesData, $batchSize);
    
            DB::beginTransaction(); // Start transaction
    
            foreach ($chunks as $chunk) {
                foreach ($chunk as &$data) {
                    $data['dtUpdatedDate'] = $currentDate; // Timestamp
                    $data['iBisActive'] = 1;              // Set active flag
                    $data['iYear'] = $year;               // Add the provided year
                }
                DB::table('tblCutOff')->insert($chunk);
            }
    
            DB::commit(); // Commit transaction
    
            return response()->json(['result' => '2', 'message' => 'Cut Off processed successfully.'], 201);
    
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on error
            \Log::error('Error in addMultipleCutOff:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json(['result' => '0', 'message' => 'An error occurred while processing Cut Off.'], 500);
        }
}


    public function addCutOff(Request $request)
{
    // Validate the request (you can expand validation rules as needed)
    $validated = $request->validate([
        'iCollegeCode'        => 'required|integer',
        'sInstituteUniqueID'  => 'nullable|string',
        'sRound'              => 'required|string',
        'sCourseName'         => 'nullable|string',
        'sQuota'              => 'nullable|string',
        'sSeatType'           => 'nullable|string',
        'iAIR'                => 'nullable|integer',
        'iSML'                => 'nullable|integer',
        'iNeetMarks'          => 'nullable|numeric',
        'sProgram'            => 'required|string',
        'sBranchName'         => 'nullable|string',
        'sPercentile'         => 'nullable|string',
        'iYear'               => 'required|integer',
        'iBranchCode'         => 'nullable|string',
        'iBisActive'          => 'nullable|boolean',
        'dtUpdatedDate'       => 'nullable|date',
    ]);

    // Insert into the database
    $id = DB::table('tblCutOff')->insertGetId([
        'iCollegeCode'        => $validated['iCollegeCode'],
        'sInstituteUniqueID'  => $validated['sInstituteUniqueID'?? null],
        'sRound'              => $validated['sRound'],
        'sCourseName'         => $validated['sCourseName'],
        'sQuota'              => $validated['sQuota']?? null,
        'sSeatType'           => $validated['sSeatType']?? null,
        'iAIR'                => $validated['iAIR'] ?? null,
        'iSML'                => $validated['iSML'] ?? null,
        'iNeetMarks'          => $validated['iNeetMarks'] ?? null,
        'sProgram'            => $validated['sProgram'],
        'iBranchCode'         => $validated['iBranchCode']??null,
        'sBranchName'         => $validated['sBranchName']?? null,
        'sPercentile'         => $validated['sPercentile'] ?? null,
        'iYear'               => $validated['iYear'],
        'iBisActive'          => $validated['iBisActive']?? 1,
        'dtUpdatedDate'       => $validated['dtUpdatedDate'] ?? now(),
        
    ]);

    // Return response
    return response()->json([
        'result'  => "1",
        'message' => 'Cutoff added successfully',
        'id'      => $id
    ]);
}




    // Update an existing record
    public function updateCutOffList(Request $request)
    {
        $data = $request->only([
           'iCollegeCode', 'sInstituteUniqueID', 'sRound', 
            'sCourseName', 'sQuota','sSeatType','iAIR', 'iSML', 'sUniversityName', 
            'sAutonomyStatus', 'sGendorEducation', 'sMinorityStatus', 'iNeetMarks', 'sProgram','sBranchName','sRound','sPercentile','iYear','iBisActive', 
            'dtUpdatedDate'
        ]);

        $iId = $request->input('iId');
        $data['dtUpdatedDate']=now();
        $updated = DB::table('tblCutOff')->where('iId', $iId)->update($data);

        if ($updated) {
            return response()->json(['result'=>'3','message' => 'Institute Cut-off updated']);
        } else {
            return response()->json(['message' => 'Institute Cut-off not found'], 404);
        }
    }



    // Delete a record
    public function deleteCutOff($iId)
    {
        $deleted = DB::table('tblCutOff')->where('iId', $iId)->update(['iBisActive' => 0]);
        if ($deleted) {
            return response()->json(['result'=>'2','message' => 'Institute Cut-off deleted']);
        } else {
            return response()->json(['message' => 'Institute Cut-off not found'], 404);
        }
    }
    




    // get cutofflist by pagination
    
    
    //     public function getAllCutOffListbyPage(Request $request)
    //     {
    //     try {
    //         $validatedData = $request->validate([
    //             'per_page' => 'integer|nullable',
    //             'current_page' => 'integer|nullable',
    //         ]);
    
    //         $perPage = $validatedData['per_page'] ?? 10;
    //         $currentPage = $validatedData['current_page'] ?? 0;
    
    //         // Subquery to deduplicate tblinstitutedetails and include other fields
    //         $subQuery = DB::table('tblinstitutedetails')
    //             ->select(
    //                 'iCollegeCode',
    //                 DB::raw('MAX(sCollegeName) as sCollegeName'),
    //                 DB::raw('MAX(sStatus) as sStatus'),
    //                 DB::raw('MAX(sDistrict) as sDistrict'),
    //                 DB::raw('MAX(sUniversityName) as sUniversityName'),
    //                 DB::raw('MAX(sAutonomyStatus) as sAutonomyStatus'),
    //                 DB::raw('MAX(sIntake) as sIntake'),
    //                 DB::raw('MAX(sPhoneNo) as sPhoneNo'),
    //                 DB::raw('MAX(imageURL) as imageURL'),
    //                 DB::raw('MAX(sAddress) as sAddress')
    //             )
    //             ->groupBy('iCollegeCode');
    
    //         // Main query with left join
    //         $query = DB::table('tblCutOff')
    //             ->leftJoinSub($subQuery, 'tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
    //             ->select(
    //                 'tblCutOff.*',
    //                 'tblinstitutedetails.sCollegeName',
    //                 'tblinstitutedetails.sStatus',
    //                 'tblinstitutedetails.sDistrict',
    //                 'tblinstitutedetails.sUniversityName',
    //                 'tblinstitutedetails.sAutonomyStatus',
    //                 'tblinstitutedetails.sIntake',
    //                 'tblinstitutedetails.sPhoneNo',
    //                 'tblinstitutedetails.imageURL',
    //                 'tblinstitutedetails.sAddress'
    //             )
    //             ->groupBy('tblCutOff.sProgram');
    
    //         // Pagination logic
    //         $offset = ($currentPage - 1) * $perPage;
    //         $total = $query->count();
    //         $colleges = $query->offset($offset)->limit($perPage)->get();
    
    //         $lastPage = (int) ceil($total / $perPage);
    
    //         // Return paginated response
    //         return response()->json([
    //             'result' => '1',
    //             'success' => true,
    //             'data' => $colleges,
    //             'current_page' => $currentPage,
    //             'last_page' => $lastPage,
    //             'total' => $total,
    //             'per_page' => $perPage,
    //             'message' => 'Colleges retrieved successfully.',
    //         ]);
    //     } catch (\Exception $e) {
    //         \Log::error('Error in getAllCutOffListbyPage:', [
    //             'error' => $e->getMessage(),
    //             'trace' => $e->getTraceAsString(),
    //         ]);
    
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'An error occurred while retrieving colleges.',
    //         ], 500);
    //     }
    // }

    public function getAllCutOffListbyPage(Request $request)
    {
    try {
        $validatedData = $request->validate([
            'per_page' => 'integer|nullable',
            'current_page' => 'integer|nullable',
        ]);

        $perPage = $validatedData['per_page'] ?? 10;
        $currentPage = $validatedData['current_page'] ?? 1;  // fixed: page 0 would skip everything

        // Subquery to deduplicate tblinstitutedetails
        $subQuery = DB::table('tblinstitutedetails')
            ->select(
                'iCollegeCode',
                DB::raw('MAX(sCollegeName) as sCollegeName'),
                DB::raw('MAX(sStatus) as sStatus'),
                DB::raw('MAX(sDistrict) as sDistrict'),
                DB::raw('MAX(sUniversityName) as sUniversityName'),
                DB::raw('MAX(sAutonomyStatus) as sAutonomyStatus'),
                DB::raw('MAX(sIntake) as sIntake'),
                DB::raw('MAX(sPhoneNo) as sPhoneNo'),
                DB::raw('MAX(imageURL) as imageURL'),
                DB::raw('MAX(sAddress) as sAddress')
            )
            ->groupBy('iCollegeCode');

        // Main query with left join — NO GROUP BY
        $query = DB::table('tblCutOff')
            ->leftJoinSub($subQuery, 'tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
            ->select(
                'tblCutOff.*',
                'tblinstitutedetails.sCollegeName',
                'tblinstitutedetails.sStatus',
                'tblinstitutedetails.sDistrict',
                'tblinstitutedetails.sUniversityName',
                'tblinstitutedetails.sAutonomyStatus',
                'tblinstitutedetails.sIntake',
                'tblinstitutedetails.sPhoneNo',
                'tblinstitutedetails.imageURL',
                'tblinstitutedetails.sAddress'
            );

        // Pagination logic
        $offset = ($currentPage - 1) * $perPage;
        $total = $query->count();

        $colleges = $query->offset($offset)->limit($perPage)->get();

        $lastPage = (int) ceil($total / $perPage);

        return response()->json([
            'result' => '1',
            'success' => true,
            'data' => $colleges,
            'current_page' => $currentPage,
            'last_page' => $lastPage,
            'total' => $total,
            'per_page' => $perPage,
            'message' => 'Colleges retrieved successfully.',
        ]);
    } catch (\Exception $e) {
        \Log::error('Error in getAllCutOffListbyPage:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while retrieving colleges.',
        ], 500);
    }
}

    
    
    
    // get cut off list by filter
    public function getAllCutOffByFilter(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'query' => 'nullable|string',
                'per_page' => 'integer|nullable',
                'current_page' => 'integer|nullable',
                'sortField' => 'nullable|string',
                'sortOrder' => 'nullable|string',
                'filters' => 'nullable|array',
            ]);
    
            $perPage = $validatedData['per_page'] ?? 10;
            $currentPage = $validatedData['current_page'] ?? 1;
            $filters = $validatedData['filters'] ?? [];
            $query = $validatedData['query'] ?? null;
            $sortField = $validatedData['sortField'] ?? 'c.iId';
            $sortOrder = $validatedData['sortOrder'] ?? 'asc';
    
            // Get MIN(iId) per iCollegeCode + sProgram
            $minIdsQuery = DB::table('tblCutOff')
                ->select(DB::raw('MIN(iId) as iId'))
                ->where('iBisActive', 1)
                ->groupBy('iCollegeCode', 'sProgram' ,'sCourseName');
    
            if (!empty($filters['program'])) {
                $minIdsQuery->whereIn('sProgram', $filters['program']);
            }
            if (!empty($filters['course'])) {
                $minIdsQuery->whereIn('sCourseName', $filters['course']);
            }
            if (!empty($filters['branch'])) {
                $minIdsQuery->whereIn('sBranchName', $filters['branch']);
            }
            if (!empty($filters['round'])) {
                $minIdsQuery->whereIn('sRound', $filters['round']);
            }
            if (!empty($filters['air'])) {
                $minIdsQuery->where(function ($q) use ($filters) {
                    foreach ($filters['air'] as $range) {
                        [$min, $max] = explode('-', $range);
                        $q->orWhereBetween('iAIR', [(int)$min, (int)$max]);
                    }
                });
            }
            if (!empty($filters['sml'])) {
                $minIdsQuery->where(function ($q) use ($filters) {
                    foreach ($filters['sml'] as $range) {
                        [$min, $max] = explode('-', $range);
                        $q->orWhereBetween('iSML', [(int)$min, (int)$max]);
                    }
                });
            }
    
            // Join filtered MIN(iId) records with latest institute data per (iCollegeCode, sProgram)
            $cutOffQuery = DB::table('tblCutOff as c')
                ->joinSub($minIdsQuery, 'filtered', function ($join) {
                    $join->on('c.iId', '=', 'filtered.iId');
                })
                ->leftJoin(DB::raw('
                    (SELECT 
                        MAX(iId) as iId,
                        iCollegeCode,
                        sProgram
                     FROM tblinstitutedetails
                     GROUP BY iCollegeCode, sProgram
                    ) as i_latest
                '), function ($join) {
                    $join->on('c.iCollegeCode', '=', 'i_latest.iCollegeCode')
                         ->on('c.sProgram', '=', 'i_latest.sProgram');
                })
                ->leftJoin('tblinstitutedetails as i', 'i.iId', '=', 'i_latest.iId')
                ->select(
                    'c.*',
                    'i.sCollegeName',
                    'i.sStatus as institute_status',
                    'i.sDistrict',
                    'i.sUniversityName',
                    'i.sAutonomyStatus',
                    'i.sIntake',
                    'i.sPhoneNo',
                    'i.imageURL',
                    'i.sAddress',
                    'i.sState'
                );
    
            // Apply state/city/status filters
            if (!empty($filters['state'])) {
                $cutOffQuery->whereIn('i.sState', $filters['state']);
            }
            if (!empty($filters['city'])) {
                $cutOffQuery->whereIn('i.sDistrict', $filters['city']);
            }
            if (!empty($filters['status'])) {
                $cutOffQuery->whereIn('i.sStatus', $filters['status']);
            }
    
            // Apply search query
            if (!empty($query)) {
                $cutOffQuery->where(function ($q) use ($query) {
                    $q->orWhere('c.sProgram', 'like', "%$query%")
                      ->orWhere('c.sCourseName', 'like', "%$query%")
                      ->orWhere('i.sCollegeName', 'like', "%$query%")
                      ->orWhere('i.sUniversityName', 'like', "%$query%")
                      ->orWhere('i.iCollegeCode', 'like', "%$query%");
                });
            }
    
            // Apply sorting
            $cutOffQuery->orderBy($sortField, $sortOrder);
    
            // Count and paginate
            $total = $cutOffQuery->count();
            $result = $cutOffQuery->offset(($currentPage - 1) * $perPage)->limit($perPage)->get();
    
            return response()->json([
                'success' => true,
                'data' => $result,
                'current_page' => $currentPage,
                'last_page' => ceil($total / $perPage),
                'total' => $total,
                'per_page' => $perPage,
                'message' => 'Distinct cutoff data retrieved.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getAllCutOffByFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving data.',
            ], 500);
        }
    }




    
    // get distinct states
    public function getStatesByCutOff(Request $request)
    {
        try {
            $request->validate([
                'program' => 'nullable|array',
                'program.*' => 'string',
            ]);
    
            $programs = $request->input('program', []);
    
            $query = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                ->select(
                    'tblinstitutedetails.sState as state',
                    DB::raw('COUNT(DISTINCT tblCutOff.iCollegeCode) AS count')
                )
                ->where('tblCutOff.iBisActive', 1)
                ->where('tblCutOff.iYear', 2025)
                ->whereNotNull('tblinstitutedetails.sState')
                ->groupBy('tblinstitutedetails.sState');
    
            if (!empty($programs)) {
                $query->whereIn('tblCutOff.sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json($results);
        } catch (\Exception $e) {
            \Log::error('Error in getStatesByCutOff:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving states.',
            ], 500);
        }
    }






    // get distinct cities
  public function getDistrictsByState(Request $request)
  {
    try {
        // Validate request
        $request->validate([
            'states' => 'nullable|array',
            'states.*' => 'string',
            'program' => 'nullable|array',
            'program.*' => 'string',
        ]);

        $states = $request->input('states', []);
        $programs = $request->input('program', []);

        $query = DB::table('tblCutOff')
            ->leftJoin('tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
            ->select(
                'tblinstitutedetails.sState as state',
                'tblinstitutedetails.sDistrict as district',
                DB::raw('COUNT(DISTINCT tblCutOff.iCollegeCode) as count')
            )
            ->where('tblCutOff.iBisActive', 1)
            ->whereNotNull('tblinstitutedetails.sDistrict')
            ->groupBy('tblinstitutedetails.sState', 'tblinstitutedetails.sDistrict');

        if (!empty($states)) {
            $query->whereIn('tblinstitutedetails.sState', $states);
        }

        if (!empty($programs)) {
            $query->whereIn('tblCutOff.sProgram', $programs);
        }

        $result = $query->get();

        return response()->json([
            'success' => true,
            'data' => $result,
            'message' => 'Districts by state retrieved successfully.',
        ]);
    } catch (\Exception $e) {
        \Log::error('Error in getDistrictsByState:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while retrieving districts by state.',
        ], 500);
    }
}


    



    // get distinct programs
     public function getProgramsByCutOffFilter1(Request $request)
    {
        try {
            $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedCities.*' => 'string', 
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedStates.*' => 'string',
            ]);
    
            $cities = $request->input('filter.selectedCities', []); 
            $states = $request->input('filter.selectedStates', []);
    
            $query = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                ->select(
                    'tblCutOff.sProgram',
                    DB::raw('COUNT(DISTINCT tblCutOff.iCollegeCode) AS count')
                )
                ->where('tblCutOff.iBisActive', 1)
                ->whereNotNull('tblCutOff.sProgram')
                ->groupBy('tblCutOff.sProgram');
    
            if (!empty($states)) {
                $query->whereIn('tblinstitutedetails.sState', $states);
            }
    
            if (!empty($cities)) {
                $query->whereIn('tblinstitutedetails.sDistrict', $cities);
            }
    
            $result = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $result,
                'message' => 'Programs retrieved successfully based on the filters.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getProgramsByCutOffFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving programs.',
            ], 500);
        }
    }

    

    // get  courses by filter
      public function getCoursesByCutOffFilter(Request $request)
      {
        try {
            $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedCities.*' => 'string', 
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedStates.*' => 'string', 
                'filter.selectedPrograms' => 'nullable|array',
                'filter.selectedPrograms.*' => 'string', 
            ]);
    
            // Extract filters
            $cities = $request->input('filter.selectedCities', []);
            $states = $request->input('filter.selectedStates', []); 
            $programs = $request->input('filter.selectedPrograms', []); 
    
            $query = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                ->select(
                    'tblCutOff.sCourseName',
                    DB::raw('COUNT(DISTINCT tblCutOff.iCollegeCode) AS count')
                )
                ->where('tblCutOff.iBisActive', 1)
                ->whereNotNull('tblCutOff.sCourseName')
                ->groupBy('tblCutOff.sCourseName');
    
            if (!empty($states)) {
                $query->whereIn('tblinstitutedetails.sState', $states); 
            }
    
            if (!empty($cities)) {
                $query->whereIn('tblinstitutedetails.sDistrict', $cities); 
            }
    
            if (!empty($programs)) {
                $query->whereIn('tblCutOff.sProgram', $programs); 
            }
    
            $result = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $result,
                'message' => 'Courses retrieved successfully based on the filters.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getCoursesByCutOffFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving courses.',
            ], 500);
        }
    }



    // get branch by filter
     public function getBranchByCutOffFilter(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedCities.*' => 'string',
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedStates.*' => 'string',
                'filter.selectedPrograms' => 'nullable|array',
                'filter.selectedPrograms.*' => 'string',
                'filter.selectCategory' => 'nullable|array',
                'filter.selectCategory.*' => 'string',
            ]);
    
            // Extract filters
            $cities = $request->input('filter.selectedCities', []);
            $states = $request->input('filter.selectedStates', []);
            $programs = $request->input('filter.selectedPrograms', []);
            $course = $request->input('filter.selectCategory', []);
    
            // Build the base query
            $query = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                ->select(
                    'tblCutOff.sBranchName',
                    DB::raw('COUNT(DISTINCT tblCutOff.iCollegeCode) AS count'),
                    DB::raw('MIN(tblCutOff.iAIR) as min_air'),
                    DB::raw('MAX(tblCutOff.iAIR) as max_air'),
                    DB::raw('MIN(tblCutOff.iSML) as min_sml'),
                    DB::raw('MAX(tblCutOff.iSML) as max_sml')
                )
                ->where('tblCutOff.iBisActive', 1)
                ->whereNotNull('tblCutOff.sBranchName')
                ->groupBy('tblCutOff.sBranchName');
    
            // Apply filters
            if (!empty($states)) {
                $query->whereIn('tblinstitutedetails.sState', $states);
            }
    
            if (!empty($cities)) {
                $query->whereIn('tblinstitutedetails.sDistrict', $cities);
            }
    
            if (!empty($programs)) {
                $query->whereIn('tblCutOff.sProgram', $programs);
            }
    
            if (!empty($course)) {
                $query->whereIn('tblCutOff.sCourseName', $course);
            }
    
            // Execute the query
            $result = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $result,
                'total' => $result->count(),
                'message' => 'Branches retrieved successfully based on the filters.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getBranchByCutOffFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving branches.',
            ], 500);
        }
    }
 




     // get  status by filter
    public function getStatusByCutOffFilter(Request $request)
    {
        try {
            $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedCities.*' => 'string', 
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedStates.*' => 'string', 
                'filter.selectedPrograms' => 'nullable|array',
                'filter.selectedPrograms.*' => 'string', 
                'filter.selectCategory' => 'nullable|array',
                'filter.selectCategory.*' => 'string', 
            ]);
    
            // Extract filters
            $cities = $request->input('filter.selectedCities', []);
            $states = $request->input('filter.selectedStates', []);
            $programs = $request->input('filter.selectedPrograms', []);
            $course = $request->input('filter.selectCategory', []);
    
            // Build the query
            $query = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode') 
                ->select(
                    'tblinstitutedetails.sStatus',
                    DB::raw('COUNT(DISTINCT tblCutOff.iCollegeCode) AS count')
                )
                ->where('tblCutOff.iBisActive', 1)
                ->whereNotNull('tblinstitutedetails.sStatus')
                ->groupBy('tblinstitutedetails.sStatus');
    
            // Apply filters
            if (!empty($states)) {
                $query->whereIn('tblinstitutedetails.sState', $states);
            }
    
            if (!empty($cities)) {
                $query->whereIn('tblinstitutedetails.sDistrict', $cities);
            }
    
            if (!empty($programs)) {
                $query->whereIn('tblCutOff.sProgram', $programs);
            }
    
            if (!empty($course)) {
                $query->whereIn('tblCutOff.sCourseName', $course);
            }
    
            // Execute
            $result = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $result, 
                'message' => 'Statuses retrieved successfully based on the filters.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getStatusByCutOffFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(), 
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving statuses.',
            ], 500);
        }
    }





      // get  status by filter
     public function getRoundByCutOffFilter(Request $request)
    {
        try {
            // Validate incoming request
            $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedCities.*' => 'string', 
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedStates.*' => 'string', 
                'filter.selectedPrograms' => 'nullable|array',
                'filter.selectedPrograms.*' => 'string', 
                'filter.selectCategory' => 'nullable|array',
                'filter.selectCategory.*' => 'string', 
                'filter.selectStatus' => 'nullable|array',
                'filter.selectStatus.*' => 'string', 
            ]);
    
            // Extract filters from the request
            $cities = $request->input('filter.selectedCities', []);
            $states = $request->input('filter.selectedStates', []); 
            $programs = $request->input('filter.selectedPrograms', []); 
            $courses = $request->input('filter.selectCategory', []); 
            $status = $request->input('filter.selectStatus', []); 
    
            // Base query
            $query = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', function ($join) {
                    $join->on('tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                         ->on('tblCutOff.sProgram', '=', 'tblinstitutedetails.sProgram');
                         // Removed the sCourseName = sBranchName join condition
                })
                ->select(
                    'tblCutOff.sRound',
                    DB::raw('COUNT(DISTINCT tblCutOff.iCollegeCode) AS count')
                )
                ->where('tblCutOff.iBisActive', 1)
                ->whereNotNull('tblCutOff.sRound');
    
            // Apply filters
            if (!empty($states)) {
                $query->whereIn('tblinstitutedetails.sState', $states);
            }
    
            if (!empty($cities)) {
                $query->whereIn('tblinstitutedetails.sDistrict', $cities);
            }
    
            if (!empty($programs)) {
                $query->whereIn('tblCutOff.sProgram', $programs);
            }
    
            if (!empty($courses)) {
                $query->whereIn('tblCutOff.sCourseName', $courses);
            }
    
            if (!empty($status)) {
                $query->whereIn('tblinstitutedetails.sStatus', $status);
            }
    
            // Execute and group result
            $result = $query->groupBy('tblCutOff.sRound')->get();
    
            return response()->json([
                'success' => true,
                'data' => $result, 
                'message' => 'Rounds retrieved successfully based on the filters.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getRoundByCutOffFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(), 
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving rounds.',
            ], 500);
        }
    }

      
      



    // get AIR rank by filter  
    public function getAirRangeByCutOffFilter(Request $request)
    {
        try {
            $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedPrograms' => 'nullable|array',
                'filter.selectCategory' => 'nullable|array',
                'filter.selectStatus' => 'nullable|array',
                'filter.selectRound' => 'nullable|array',
            ]);
    
            // Extract filters
            $cities = $request->input('filter.selectedCities', []);
            $states = $request->input('filter.selectedStates', []);
            $programs = $request->input('filter.selectedPrograms', []);
            $course = $request->input('filter.selectCategory', []);
            $status = $request->input('filter.selectStatus', []);
            $round = $request->input('filter.selectRound', []);
    
            // Base Query
            $filteredQuery = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', function ($join) {
                    $join->on('tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                         ->on('tblCutOff.sProgram', '=', 'tblinstitutedetails.sProgram');
                    // ✅ sCourseName = sBranchName removed
                })
                ->where('tblCutOff.iBisActive', 1)
                ->whereNotNull('tblCutOff.iAIR');
    
            // Filters
            if (!empty($states)) {
                $filteredQuery->whereIn('tblinstitutedetails.sState', $states);
            }
            if (!empty($cities)) {
                $filteredQuery->whereIn('tblinstitutedetails.sDistrict', $cities);
            }
            if (!empty($programs)) {
                $filteredQuery->whereIn('tblCutOff.sProgram', $programs);
            }
            if (!empty($course)) {
                $filteredQuery->whereIn('tblCutOff.sCourseName', $course);
            }
            if (!empty($status)) {
                $filteredQuery->whereIn('tblinstitutedetails.sStatus', $status);
            }
            if (!empty($round)) {
                $filteredQuery->whereIn('tblCutOff.sRound', $round);
            }
    
            // Get min and max AIR
            $minMaxResult = $filteredQuery->select(
                DB::raw('MIN(tblCutOff.iAIR) as minAir'),
                DB::raw('MAX(tblCutOff.iAIR) as maxAir')
            )->first();
    
            if (!$minMaxResult || $minMaxResult->minAir === null || $minMaxResult->maxAir === null) {
                return response()->json([
                    'success' => false,
                    'message' => 'No AIR data found for the given filters.',
                    'result' => '5'
                ]);
            }
    
            $rangeInterval = 1000;
            $minAir = (int) $minMaxResult->minAir;
            $maxAir = (int) $minMaxResult->maxAir;
    
            // Generate AIR Ranges
            $ranges = [];
            for ($start = $minAir; $start <= $maxAir; $start += $rangeInterval) {
                $end = min($start + $rangeInterval - 1, $maxAir);
                $ranges[] = [
                    'start' => $start,
                    'end' => $end,
                    'count' => 0,
                    'range' => "$start - $end",
                ];
            }
    
            // Get count in each AIR range
            $airCounts = $filteredQuery
                ->select(
                    DB::raw("FLOOR(tblCutOff.iAIR / $rangeInterval) * $rangeInterval as air_range"),
                    DB::raw('COUNT(DISTINCT tblCutOff.iId) as count')
                )
                ->groupBy(DB::raw("FLOOR(tblCutOff.iAIR / $rangeInterval) * $rangeInterval"))
                ->get();
    
            // Map counts
            foreach ($airCounts as $air) {
                foreach ($ranges as &$range) {
                    if ($air->air_range >= $range['start'] && $air->air_range <= $range['end']) {
                        $range['count'] = $air->count;
                        break;
                    }
                }
            };
    
$filteredRanges = array_filter($ranges, function($range) {
    return $range['count'] > 0;
});
    
            return response()->json([
                'success' => true,
                'data' => array_values($filteredRanges),
                'message' => 'Filtered AIR ranges with counts retrieved successfully.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getAirRangeByCutOffFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving AIR range with count.',
            ], 500);
        }
    }

    
    
    



    // get sml rank by filter
    public function getSMLRangeByCutOffFilter(Request $request)
    {
        try {
            $filters = $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedPrograms' => 'nullable|array',
                'filter.selectCategory' => 'nullable|array',
                'filter.selectStatus' => 'nullable|array',
                'filter.selectRound' => 'nullable|array',
            ])['filter'] ?? [];
    
            // Base query with join (NO sCourseName = sBranchName condition)
            $baseQuery = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', function ($join) {
                    $join->on('tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                         ->on('tblCutOff.sProgram', '=', 'tblinstitutedetails.sProgram');
                })
                ->where('tblCutOff.iBisActive', 1)
                ->whereNotNull('tblCutOff.iSML');
    
            // Apply filters dynamically
            if (!empty($filters['selectedStates'])) {
                $baseQuery->whereIn('tblinstitutedetails.sState', $filters['selectedStates']);
            }
            if (!empty($filters['selectedCities'])) {
                $baseQuery->whereIn('tblinstitutedetails.sDistrict', $filters['selectedCities']);
            }
            if (!empty($filters['selectedPrograms'])) {
                $baseQuery->whereIn('tblCutOff.sProgram', $filters['selectedPrograms']);
            }
            if (!empty($filters['selectCategory'])) {
                $baseQuery->whereIn('tblCutOff.sCourseName', $filters['selectCategory']);
            }
            if (!empty($filters['selectStatus'])) {
                $baseQuery->whereIn('tblinstitutedetails.sStatus', $filters['selectStatus']);
            }
            if (!empty($filters['selectRound'])) {
                $baseQuery->whereIn('tblCutOff.sRound', $filters['selectRound']);
            }
    
            // Get min & max SML (casted to int)
            $minMax = $baseQuery
                ->selectRaw('MIN(CAST(tblCutOff.iSML AS SIGNED)) as minSML, MAX(CAST(tblCutOff.iSML AS SIGNED)) as maxSML')
                ->first();
    
            if (!$minMax || is_null($minMax->minSML) || is_null($minMax->maxSML)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No SML data found.',
                    'result' => '5'
                ]);
            }
    
            $minSML = (int) $minMax->minSML;
            $maxSML = (int) $minMax->maxSML;
    
            // Ensure correct order
            if ($minSML > $maxSML) {
                [$minSML, $maxSML] = [$maxSML, $minSML];
            }
    
            // Build SML ranges with counts
            $ranges = [];
            $interval = 2000;
            for ($start = $minSML; $start <= $maxSML; $start += $interval) {
                $end = min($start + $interval - 1, $maxSML);
    
                $count = DB::table('tblCutOff')
                    ->leftJoin('tblinstitutedetails', function ($join) {
                        $join->on('tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                             ->on('tblCutOff.sProgram', '=', 'tblinstitutedetails.sProgram');
                    })
                    ->where('tblCutOff.iBisActive', 1)
                    ->whereNotNull('tblCutOff.iSML')
                    ->whereBetween(DB::raw('CAST(tblCutOff.iSML AS SIGNED)'), [$start, $end]);
    
                // Reapply same filters for count query
                if (!empty($filters['selectedStates'])) {
                    $count->whereIn('tblinstitutedetails.sState', $filters['selectedStates']);
                }
                if (!empty($filters['selectedCities'])) {
                    $count->whereIn('tblinstitutedetails.sDistrict', $filters['selectedCities']);
                }
                if (!empty($filters['selectedPrograms'])) {
                    $count->whereIn('tblCutOff.sProgram', $filters['selectedPrograms']);
                }
                if (!empty($filters['selectCategory'])) {
                    $count->whereIn('tblCutOff.sCourseName', $filters['selectCategory']);
                }
                if (!empty($filters['selectStatus'])) {
                    $count->whereIn('tblinstitutedetails.sStatus', $filters['selectStatus']);
                }
                if (!empty($filters['selectRound'])) {
                    $count->whereIn('tblCutOff.sRound', $filters['selectRound']);
                }
    
                $ranges[] = [
                    'range' => "$start - $end",
                    'start' => $start,
                    'end' => $end,
                    'count' => $count->count('tblCutOff.iId')
                ];
            }
    
            return response()->json([
                'success' => true,
                'data' => $ranges,
                'message' => 'SML ranges with cutoff counts generated successfully.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error generating SML ranges:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while generating SML ranges.',
            ], 500);
        }
    }



    // Get distinct Branch Names by college code, program, and course name
    public function getDistinctRound(Request $request)
    {
        try {
            // Validate incoming parameters
            $request->validate([
                'collegecode' => 'required|integer',
                'sProgram' => 'required|string',
                'sCourseName' => 'required|string',
            ]);
    
            // Get the request data
            $collegeCode = $request->input('collegecode');
            $program = $request->input('sProgram');
            $courseName = $request->input('sCourseName');
    
            // Query to fetch distinct sBranchName
            $branches = DB::table('tblCutOff')
                ->leftJoin('tblinstitutedetails', function ($join) {
                    $join->on('tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                        ->on('tblCutOff.sProgram', '=', 'tblinstitutedetails.sProgram');
                })
                ->where('tblCutOff.iCollegeCode', $collegeCode)
                ->where('tblCutOff.sProgram', $program)
                ->where('tblCutOff.sCourseName', $courseName)
                ->distinct()
                ->pluck('tblCutOff.sRound'); // Get distinct branch names
    
            if ($branches->isEmpty()) {
                return response()->json([
                    'result' => '5',
                    'message' => 'No branches found for the given parameters.',
                ], 404);
            }
    
            return response()->json([
                'result' => '1',
                'message' => 'Distinct Branch Names retrieved successfully.',
                'data' => $branches,
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctBranchNames:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'result' => '0',
                'message' => 'An error occurred while retrieving distinct branch names.',
            ], 500);
        }
    }



    // Get distinct Branch Names by college code, program, and course name
    public function getDistinctBranches(Request $request)
    {
        try {
            $request->validate([
                'collegecode' => 'required|integer',
                'sProgram' => 'required|string',
                'sCourseName' => 'required|string',
                'sRound' => 'nullable|string',
                'sCategory' => 'nullable|string'
            ]);
    
            $collegeCode = $request->input('collegecode');
            $program = $request->input('sProgram');
            $courseName = $request->input('sCourseName');
            $round = $request->input('sRound');
            $category = $request->input('sCategory');
    
            $branches = DB::table('tblCutOff')
                ->where('iCollegeCode', $collegeCode)
                ->where('sProgram', $program)
                ->where('sCourseName', $courseName)
                ->when(!empty($round), function($query) use ($round) {
                    return $query->where('sRound', $round);
                })
                ->when(!empty($category), function($query) use ($category) {
                    return $query->where('sSeatType', $category);
                })
                ->distinct()
                ->pluck('sBranchName');
    
            if ($branches->isEmpty()) {
                return response()->json([
                    'result' => '2',
                    'message' => 'No branches found for the given parameters.',
                ], 404);
            }
    
            return response()->json([
                'result' => '1',
                'message' => 'Distinct Branch Names retrieved successfully.',
                'data' => $branches,
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctBranches:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'result' => '0',
                'message' => 'An error occurred while retrieving distinct branch names.',
            ], 500);
        }
    }





    // get cut off with intake
    public function getCutOffWithIntake(Request $request)
    {
        try {
            //  Step 1: Validate the request
            $validated = $request->validate([
                'iCollegeCode' => 'required|integer',
                'sProgram'     => 'required|string',
                'sCourseName'  => 'required|string',
                'sBranchName'  => 'required|string',
                'sRound'       => 'nullable|string',
                'sCategory'    => 'nullable|string'
            ]);
    
            // Step 2: Enable query logging for debugging (optional)
            \DB::enableQueryLog();
    
          $result = DB::table('tblCutOff as c')
        ->select('c.*', 'i.sIntake')
        ->leftJoin('tblinstitutedetails as i', function ($join) use ($validated) {
            $join->on('c.iCollegeCode', '=', 'i.iCollegeCode')
                 ->on('c.sProgram', '=', 'i.sProgram')
                 ->on('c.sCourseName', '=', 'i.sCourseName')
                 ->on('c.sBranchName', '=', 'i.sBranchName');
        })
        ->where('c.iCollegeCode', $validated['iCollegeCode'])
        ->where('c.sProgram', $validated['sProgram'])
        ->where('c.sCourseName', $validated['sCourseName'])
        ->where('c.sBranchName', $validated['sBranchName'])
        ->when(!empty($validated['sRound']), function($query) use ($validated) {
            return $query->where('c.sRound', $validated['sRound']);
        })
         ->when(!empty($validated['sCategory']), function($query) use ($validated) {
            return $query->where('c.sSeatType', $validated['sCategory']);
        })
        ->where('c.iBisActive', 1)
        ->distinct()
        ->get();
    
            // Step 4: Log final SQL (optional for debugging)
            \Log::info('Executed Query:', DB::getQueryLog());
    
            // tep 5: Return success response
            return response()->json([
                'success' => true,
                'message' => 'Cutoff + Intake fetched successfully.',
                'data'    => $result,
            ]);
    
        } catch (\Exception $e) {
            // ✅ Log exception details for debugging
            \Log::error('Error in getCutOffWithIntake:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch data.',
            ], 500);
        }
    }




    //get distinct categories
    	
    public function getDistinctSeatType(Request $request)
    {
    $iCollegeCode = $request->input('iCollegeCode', '');
    $sProgram = $request->input('sProgram', '');

    try {
        $query = DB::table('tblCutOff')
            ->select('sSeatType')
            ->where('iBisActive', 1)
            ->where('iCollegeCode', $iCollegeCode);

        // Check if sProgram is provided and add it to the query if it is
        if (!empty($sProgram)) {
            $query->where('sProgram', $sProgram);
        }

        $results = $query->groupBy('sSeatType')->get();

        return response()->json([
            'success' => true,
            'data' => $results
        ]);

    } catch (\Exception $e) {
        \Log::error('Error in getDistinctSeatType:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while retrieving seat types.',
        ], 500);
    }
}










//New

    public function getProgramsByCutOffFilter(Request $request)
    {
        try {
            $request->validate([
                'filter.selectedCities' => 'nullable|array',
                'filter.selectedCities.*' => 'string', 
                'filter.selectedStates' => 'nullable|array',
                'filter.selectedStates.*' => 'string', 
                'filter.selectedPrograms' => 'nullable|array',
                'filter.selectedPrograms.*' => 'string', 
                'filter.selectCategory' => 'nullable|array',
                'filter.selectCategory.*' => 'string',
                'filter.selectBranch' => 'nullable|array',
                'filter.selectBranch.*' => 'string',
                'filter.selectStatus' => 'nullable|array',
                'filter.selectStatus.*' => 'string',
                'filter.selectRound' => 'nullable|array',
                'filter.selectRound.*' => 'string',
            ]);
    
            $cities = $request->input('filter.selectedCities', []); 
            $states = $request->input('filter.selectedStates', []);
            $programs = $request->input('filter.selectedPrograms', []);
            $categories = $request->input('filter.selectCategory', []);
            $branches = $request->input('filter.selectBranch', []);
            $statuses = $request->input('filter.selectStatus', []);
            $rounds = $request->input('filter.selectRound', []);
    
            $query = DB::table('tblinstitutedetails')
                ->leftJoin('tblCutOff', 'tblCutOff.iCollegeCode', '=', 'tblinstitutedetails.iCollegeCode')
                ->select(
                    'tblCutOff.sProgram',
                    DB::raw('COUNT(DISTINCT tblinstitutedetails.iCollegeCode) AS count')
                )
                ->where('tblinstitutedetails.iBisActive', 1)
                ->whereNotNull('tblinstitutedetails.sProgram')
                ->groupBy('tblCutOff.sProgram');
    
            if (!empty($states)) {
                $query->whereIn('tblinstitutedetails.sState', $states);
            }
    
            if (!empty($cities)) {
                $query->whereIn('tblinstitutedetails.sDistrict', $cities);
            }
    
            if (!empty($programs)) {
                $query->whereIn('tblinstitutedetails.sProgram', $programs);
            }
    
            if (!empty($categories)) {
                $query->whereIn('tblinstitutedetails.sCategory', $categories);
            }
    
            if (!empty($branches)) {
                $query->whereIn('tblinstitutedetails.sBranchName', $branches);
            }
    
            if (!empty($statuses)) {
                $query->whereIn('tblinstitutedetails.sStatus', $statuses);
            }
    
            if (!empty($rounds)) {
                $query->whereIn('tblCutOff.sRound', $rounds);
            }
    
            $result = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $result,
                'message' => 'Programs retrieved successfully based on the filters.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getProgramsByCutOffFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving programs.',
            ], 500);
        }
    }
    
    
    
    public function getDistinctProgramForCutoff(Request $request)
    {
        $programs = $request->input('sProgram', []);
    
        // Fix nested array problem
        if (is_array($programs) && count($programs) === 1 && is_array($programs[0])) {
            $programs = $programs[0];
        }
    
        try {
            $query = DB::table('distinctprogram')
                ->select('*');
    
            if (!empty($programs)) {
                $query->whereIn('sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $results
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctProgram:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving program data.',
            ], 500);
        }
    }



    public function getDistinctStateForCutoff(Request $request)
    {
        $programs = $request->input('sProgram', []); // Expecting array like ['Medical', 'Engineering']
    
        try {
            $query = DB::table('distinctState') // Your distinct view
                ->select('*'); // Already distinct, so no need for distinct()

            if (!empty($programs) && is_array($programs)) {
                $query->whereIn('sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $results
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctProgram:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving program data.',
            ], 500);
        }
    }
    
    
    
     public function getDistinctDistrictForCutoff(Request $request)
    {
        $programs = $request->input('sProgram', []); // Expecting array like ['Medical', 'Engineering']
    
        try {
            $query = DB::table('distinctDistrict') // Your distinct view
                ->select('*'); // Already distinct, so no need for distinct()

            if (!empty($programs) && is_array($programs)) {
                $query->whereIn('sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $results
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctProgram:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving program data.',
            ], 500);
        }
    }
    
    
    
    
    public function getDistinctCourseForCutoff(Request $request)
    {
        $programs = $request->input('sProgram', []); // Expecting array like ['Medical', 'Engineering']
    
        try {
            $query = DB::table('distinctCourse') // Your distinct view
                ->select('*'); // Already distinct, so no need for distinct()

            if (!empty($programs) && is_array($programs)) {
                $query->whereIn('sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $results
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctProgram:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving program data.',
            ], 500);
        }
    }
    
    
    
    
     public function getDistinctBranchForCutoff(Request $request)
    {
        $programs = $request->input('sProgram', []); // Expecting array like ['Medical', 'Engineering']
    
        try {
            $query = DB::table('distinctBranch') // Your distinct view
                ->select('*'); // Already distinct, so no need for distinct()

            if (!empty($programs) && is_array($programs)) {
                $query->whereIn('sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $results
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctProgram:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving program data.',
            ], 500);
        }
    }



     public function getDistinctStatusForCutoff(Request $request)
    {
        $programs = $request->input('sProgram', []); // Expecting array like ['Medical', 'Engineering']
    
        try {
            $query = DB::table('distintStatus') // Your distinct view
                ->select('*'); // Already distinct, so no need for distinct()

            if (!empty($programs) && is_array($programs)) {
                $query->whereIn('sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $results
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctProgram:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving program data.',
            ], 500);
        }
    }
    
    
    
    
     public function getDistinctRoundForCutoff(Request $request)
    {
        $programs = $request->input('sProgram', []); // Expecting array like ['Medical', 'Engineering']
    
        try {
            $query = DB::table('distinctRound') // Your distinct view
                ->select('*'); // Already distinct, so no need for distinct()

            if (!empty($programs) && is_array($programs)) {
                $query->whereIn('sProgram', $programs);
            }
    
            $results = $query->get();
    
            return response()->json([
                'success' => true,
                'data' => $results
            ]);
    
        } catch (\Exception $e) {
            \Log::error('Error in getDistinctProgram:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving program data.',
            ], 500);
        }
    }

}

    
    



