<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;



class SubCoursesController extends Controller
{
    // 1. Add SubCourse
    public function addSubCourse(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'sProgram' => 'required|string|max:100',
                'sSubCourse' => 'required|string|max:100',
            ]);

            $subCourseId = DB::table('tblSubCourse')->insertGetId([
                'sProgram' => $validatedData['sProgram'],
                'sSubCourse' => $validatedData['sSubCourse'],
                'iBisActive' => 1, // Active by default
                'dtUpdatedDate' => now(), // Set current timestamp
            ]);

            return response()->json([
                'result' => "2",
                'message' => 'SubCourse added successfully',
                'id' => $subCourseId
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Error occurred while adding SubCourse',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 2. Get SubCourses (only Active)
    public function getSubCourses(Request $request)
    {
        try {
            $query = DB::table('tblSubCourse')->where('iBisActive', 1);

            if ($request->has('sProgram')) {
                $query->where('sProgram', $request->input('sProgram'));
            }

            $subCourses = $query->get();

            return response()->json([
                'result' => "2",
                'message' => 'SubCourses fetched successfully',
                'data' => $subCourses
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Error occurred while fetching SubCourses',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 3. Update dtUpdatedDate Only
    public function updateSubCourse(Request $request)
   {
    try {
        // Validate iId and ensure it exists in the table
        $validatedData = $request->validate([
            'iId' => 'required|integer|exists:tblSubCourse,iId',
        ]);

        // Prepare the data to update (exclude iId and iBisActive from the request data)
        $updateData = $request->except(['iId', 'iBisActive']);

        // Add the updated date to the data
        $updateData['dtUpdatedDate'] = now();

        // Perform the update for all the columns provided in the request
        $updated = DB::table('tblSubCourse')
            ->where('iId', $validatedData['iId'])
            ->where('iBisActive', 1)
            ->update($updateData);

        // Return response based on the update result
        if ($updated) {
            return response()->json([
                'result' => "3",
                'message' => 'SubCourse updated successfully'
            ], 200);
        } else {
            return response()->json([
                'result' => "0",
                'message' => 'No active SubCourse found to update'
            ], 404);
        }

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Validation error',
            'errors' => $e->errors()
        ], 422);
    } catch (\Exception $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Error occurred while updating SubCourse',
            'error' => $e->getMessage()
        ], 500);
    }
}


    // 4. Soft Delete SubCourse
    public function deleteSubCourse($id)
    {
        try {
            $deleted = DB::table('tblSubCourse')
                ->where('iId', $id)
                ->where('iBisActive', 1)
                ->update([
                    'iBisActive' => 0,
                    'dtUpdatedDate' => now(), // also update date on soft delete
                ]);

            if ($deleted) {
                return response()->json([
                    'result' => "2",
                    'message' => 'SubCourse deleted successfully (soft delete)'
                ], 200);
            } else {
                return response()->json([
                    'result' => "0",
                    'message' => 'No active SubCourse found to delete'
                ], 404);
            }

        } catch (\Exception $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Error occurred while deleting SubCourse',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    
    //add multiple subcourses
    public function addMultipleSubcourses(Request $request)
    {
    try {
        $subcoursesData = $request->input('data', []); // Expecting an array of subcourses
        $status = $request->input('status', 'append'); // 'addall' or 'append'

        if (empty($subcoursesData) || !is_array($subcoursesData)) {
            return response()->json(['result' => '0', 'message' => 'Invalid subcourse data provided.'], 400);
        }

        // If status is 'addall', delete all existing records
        if ($status === 'addall') {
            DB::table('tblSubCourse')->truncate();
        }

        $currentDate = now();
        $batchSize = 1800;
        $chunks = array_chunk($subcoursesData, $batchSize);

        foreach ($chunks as $chunk) {
            foreach ($chunk as &$data) {
                $data['dtUpdatedDate'] = $currentDate;
                $data['iBisActive'] = 1; // Mark as active
            }

            DB::table('tblSubCourse')->insert($chunk);
        }

        return response()->json([
            'result'  => '2',
            'message' => count($subcoursesData) . ' subcourses added successfully.',
        ], 201);

    } catch (\Exception $e) {
        \Log::error('Error in addSubcourses:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json(['result' => '0', 'message' => 'An error occurred while adding subcourses.'], 500);
    }
}



    // get subcourses by program
   public function getSubCoursesByProgram(Request $request)
{
    try {
        // Validate the 'sProgram' parameter as an array of strings
        $validatedData = $request->validate([
            'sProgram' => 'array|nullable',
            'sProgram.*' => 'string',  // Each element of the array should be a string
        ]);

        // Check if the 'sProgram' array is provided and not empty
        if (empty($validatedData['sProgram'])) {
            return response()->json([
                'result' => "0",
                'message' => 'No program provided'
            ], 400);
        }

        // Fetch subcourses where 'sProgram' is in the provided list of programs
        $subCourses = DB::table('tblSubCourse')
                        ->whereIn('sProgram', $validatedData['sProgram'])
                        ->where('iBisActive', 1)  // Only active records
                        ->get();

        if ($subCourses->isEmpty()) {
            return response()->json([
                'result' => "0",
                'message' => 'No subcourses found for the provided programs'
            ], 404);
        }

        return response()->json([
            'result' => "2",
            'message' => 'SubCourses fetched successfully',
            'data' => $subCourses
        ], 200);

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Validation error',
            'errors' => $e->errors()
        ], 422);
    } catch (\Exception $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Error occurred while fetching subcourses',
            'error' => $e->getMessage()
        ], 500);
    }
}



}

