<?php

namespace App\Http\Controllers\app;

use Illuminate\Support\Facades\Schema;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;


class appUserController extends Controller
{

    
    // Get all records from tblregistration
    public function index()
    {
        $registrations = DB::table('tblregistration')->where('iBisActive', 1)->get();
    
        if ($registrations->isEmpty()) {
            return response()->json([
                'result' => "6",
                'message' => 'No active registrations found',
            ], 404);  
        }

    
        // Return the results as JSON with success status
        return response()->json([
            'result' => "1",
            'message' => 'Active registrations retrieved successfully',
            'data' => $registrations,
        ], 200);  
    }
    


    
    // Get a specific record by ID
    public function show($id)
    {
        // Fetch data from the database
        $registration = DB::table('tblregistration as r')
            ->leftJoin('tblAppUser as u', 'r.iId', '=', 'u.iStudentId')
            ->select(
                'r.*', // All registration columns
                'u.sName' // All AppUser columns
            )
            ->where('r.iId', $id)
            ->where('r.iBisActive', 1)
            ->first();
    
        // Ensure data exists
        if ($registration) {
            // Check and properly decode JSON fields
            foreach (['iCounsellingPackageId', 'iAppPackageId'] as $field) {
                if (!empty($registration->$field) && is_string($registration->$field)) {
                    $decoded = ($registration->$field);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $registration->$field = $decoded; // Convert string to array
                    } else {
                        Log::error("JSON Decode Error for {$field}", ['value' => $registration->$field]);
                    }
                }
            }
    
            // Return correctly formatted JSON
            return response()->json([
                'result' => "1",
                'data' => $registration
            ], 200, [], JSON_UNESCAPED_UNICODE);
        } 
        
        // Return error response if data not found
        return response()->json([
            'result' => "6",
            'message' => 'Registration not found'
        ], 404);
    }
    
    


    // Add a new registration
    public function store(Request $request)
    {
        try {
            // ✅ Validate Input
            $validatedData = $request->validate([
                'sFirstName' => 'required|string|max:255',
                'sMiddleName' => 'nullable|string|max:255',
                'sLastName' => 'nullable|string|max:255',
                'sGender' => 'nullable|string|in:Male,Female,Other',
                'sCity' => 'nullable|string|max:255',
                'sStudentMobileNo' => 'nullable|string|max:15|unique:tblregistration,sStudentMobileNo',
                'sParentMobileNo' => 'nullable|string|max:15',
                'sEmailId' => 'nullable|email|max:255|unique:tblregistration,sEmailId',
                'iAmount' => 'nullable|integer|min:0',
                'iPartialAmount' => 'nullable|integer|min:0',
                'iConcession' => 'nullable|integer|min:0',
                'iTotalAmount' => 'nullable|integer|min:0',
                'iCounsellingPackageId' => 'nullable|array',
                'iCounsellingPackageId.*' => 'array',
                'iAppPackageId' => 'nullable|array',
                'iAppPackageId.*' => 'array',
                'sActiveStatus' => 'nullable|string|in:Active,Inactive',
                'sOtherDetails1' => 'nullable|string|max:255',
                'sOtherDetails2' => 'nullable|string|max:255',
                'sOtherDetails3' => 'nullable|string|max:255',
                'sOtherDetails4' => 'nullable|string|max:255',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'result' => 5,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }

        // ✅ Generate a 6-digit numeric password
        $generatedPassword = str_pad(random_int(100000, 999999), 6, '0', STR_PAD_LEFT);

        // ✅ Hash password using Laravel's password_hash equivalent (bcrypt)
        $hashedPassword = Hash::make($generatedPassword);

        // ✅ Prepare registration data
        $registrationData = [
            'sPassword' => $hashedPassword,  // ✅ Securely hashed password
            'Salt' => '', // ✅ Not needed with bcrypt
            'sFirstName' => $validatedData['sFirstName'] ?? null,
            'sMiddleName' => $validatedData['sMiddleName'] ?? null,
            'sLastName' => $validatedData['sLastName'] ?? null,
            'sGender' => $validatedData['sGender'] ?? null,
            'sCity' => $validatedData['sCity'] ?? null,
            'sStudentMobileNo' => $validatedData['sStudentMobileNo'] ?? null,
            'sParentMobileNo' => $validatedData['sParentMobileNo'] ?? null,
            'sEmailId' => $validatedData['sEmailId'] ?? null,
            'iAmount' => $validatedData['iAmount'] ?? 0,
            'iPartialAmount' => $validatedData['iPartialAmount'] ?? 0,
            'iConcession' => $validatedData['iConcession'] ?? 0,
            'iTotalAmount' => $validatedData['iTotalAmount'] ?? 0,
            'iCounsellingPackageId' => json_encode($validatedData['iCounsellingPackageId'] ?? []),
            'iAppPackageId' => json_encode($validatedData['iAppPackageId'] ?? []),
            'sActiveStatus' => $validatedData['sActiveStatus'] ?? 'Inactive',
            'iBisActive' => 1,
            'dtUpdatedDate' => now(),
            'sOtherDetails1' => $validatedData['sOtherDetails1'] ?? null,
            'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
            'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
            'sOtherDetails4' => $validatedData['sOtherDetails4'] ?? null,
        ];

        try {
            DB::beginTransaction();

            // ✅ Insert into tblregistration
            $registrationId = DB::table('tblregistration')->insertGetId($registrationData);

            // ✅ Insert into tblAppUser
            DB::table('tblAppUser')->insert([
                'iStudentid' => $registrationId,
                'sName' => $validatedData['sFirstName'],
                'iBisActive' => 1,
                'dtUpdatedDate' => now(),
            ]);

            DB::commit();

            return response()->json([
                'result' => 8,
                'message' => 'Registration and profile created successfully',
                'id' => $registrationId,
                'password' => $generatedPassword, // ✅ Send the raw password to frontend
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'result' => 0,
                'message' => 'Error occurred while creating registration',
                'error' => $e->getMessage()
            ], 500);
        }
    }




    // Update a registration by iId
    public function update(Request $request)
    {
        // Ensure iId is present in the request
        $iId = $request->input('iId');
        if (!$iId) {
            return response()->json(['result' => "1", 'message' => 'iId is required'], 400);
        }
    
        // Validate the incoming request data
        $validatedData = $request->validate([
            'sPassword' => 'required|string|max:255',
            'sFirstName' => 'required|string|max:255',
            'sMiddleName' => 'nullable|string|max:255',
            'sLastName' => 'required|string|max:255',
            'sGender' => 'required|string|in:Male,Female,other',
            'sCity' => 'nullable|string|max:255',
            'sEmailId' => 'nullable|email|max:255',
            'sOtherDetails1' => 'nullable|string|max:255',
            'sOtherDetails2' => 'nullable|string|max:255',
            'sOtherDetails3' => 'nullable|string|max:255',
            'sOtherDetails4' => 'nullable|string|max:255',
        ]);
    
        // Filter out null values
        $updateData = array_filter($validatedData, function ($value) {
            return !is_null($value);
        });
    
        // Convert array fields to JSON
        $arrayFields = ['sCounsellingPackage', 'sAppPackage', 'sProgram', 'sCourseName'];
        foreach ($arrayFields as $field) {
            if (isset($updateData[$field])) {
                $updateData[$field] = json_encode($updateData[$field]);
            }
        }
    
        // Ensure dtUpdatedDate is updated to current time
        $updateData['dtUpdatedDate'] = now();
    
        // Update the registration in the database
        try {
            $updated = DB::table('tblregistration')->where('iId', $iId)->update($updateData);
    
            if ($updated) {
                return response()->json(['result' => "3", 'message' => 'Registration updated successfully']);
            } else {
                return response()->json(['result' => "5", 'message' => 'No changes made or record not found'], 404);
            }
        } catch (\Exception $e) {
            return response()->json(['result' => "0", 'message' => 'Error updating registration', 'error' => $e->getMessage()], 500);
        }
    }
    
    
    

    // Delete a registration by ID
    public function destroy($iId)
    {
        $registration = DB::table('tblregistration')->where('iId', $iId)->first();

        if (!$registration) {
            return response()->json(['result' => 0, 'message' => 'Registration not found'], 404);
        }

        $iStudentId = $registration->iId;

        DB::table('tblregistration')->where('iId', $iId)->update(['iBisActive' => 0]);
        DB::table('tblqualification')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);
        DB::table('tblcategory')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);
        DB::table('tblsubscription')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);
        DB::table('tbldetails')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);

        return response()->json(['result' => 1, 'message' => 'All related records marked as inactive successfully'], 200);
    }




    // get filtered registrationa
   public function getFilteredRegistrations(Request $request)
    {
        $program = $request->input('program'); // Get program from request

        if (!$program) {
            return response()->json([
                'result' => 0,
                'message' => 'Program name is required'
            ], 400);
        }

        // Fetch records where JSON contains the matching sProgram
        $results = DB::table('tblregistration')
            ->whereRaw("JSON_CONTAINS(iCounsellingPackageId, JSON_OBJECT('sProgram', ?))", [$program])
            ->orWhereRaw("JSON_CONTAINS(iAppPackageId, JSON_OBJECT('sProgram', ?))", [$program])
            ->get();

        return response()->json([
            'result' => 1,
            'message' => 'Filtered records fetched successfully',
            'data' => $results
        ]);
    }

   

    // login
   public function login(Request $request)
   {
       $validatedData = $request->validate([
           'mobileNo' => 'required|string|max:15',  // Accepting either student or parent mobile number
           'sPassword' => 'required|string|min:4',
           'iDeviceId' => 'required|string',
       ]);
   
       $mobileNo = $validatedData['mobileNo'];
       $sPassword = $validatedData['sPassword'];
   
       // Check if the entered number matches either sStudentMobileNo or sParentMobileNo
       $user = DB::table('tblregistration')
           ->where(function ($query) use ($mobileNo) {
               $query->where('sStudentMobileNo', $mobileNo)
                     ->orWhere('sParentMobileNo', $mobileNo);
           })
           ->where('iBisActive', 1) // User must be active
           ->where('sActiveStatus', 'Active') // User status must be active
           ->first();
   
       if ($user && Hash::check($sPassword, $user->sPassword)) {
           $subscription = DB::table('tblSubscription')
               ->where('iStudentId', $user->iId)
               ->first();
   
           $output = [
               'result' => "8",
               'message' => "Login successful.",
               'records' => [
                   'sStudentMobileNo' => $user->sStudentMobileNo,
                   'sParentMobileNo' => $user->sParentMobileNo,
                   'iId' => $user->iId,
                   'sFirstName' => $user->sFirstName,
                   'iAmount' => $user->iAmount,
                   'iCounsellingPackageId' => $user->iCounsellingPackageId,
                   'iAppPackageId' => $user->iAppPackageId,
                   'sFlag' => $user->sFlag,
                   'sActiveStatus' => $user->sActiveStatus
               ],
               'subscription' => $subscription ? [
                   'sStatus' => $subscription->sStatus,
                   'sFlag' => $subscription->sFlag
               ] : [
                   'sStatus' => 'pending',
                   'sFlag' => 'NoSubscription'
               ]
           ];
       } else {
           $output = [
               'result' => "10",
               'message' => "Invalid credentials or account inactive."
           ];
       }
   
       return response()->json($output);
   }
   



    //  Forget Password
   public function forgetPassword(Request $request)
   {
       $validatedData = $request->validate([
           'sStudentMobileNo' => 'required|string|max:15',
       ]);

       $sStudentMobileNo = $validatedData['sStudentMobileNo'];
       $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

       if ($user) {
           $otp = rand(100000, 999999);
           DB::table('tblregistration')
               ->where('sStudentMobileNo', $sStudentMobileNo)
               ->update(['sOTP' => $otp]);

           return response()->json(['result' => "8", 'otp' => $otp]);
       }

       return response()->json(['result' => "5", 'message' => "Mobile number not found."]);
   }


    //Reset Password
   public function resetPassword(Request $request)
   {
       $validatedData = $request->validate([
           'sStudentMobileNo' => 'required|string|max:15',
           'sOTP' => 'required|string',
           'sPassword' => 'required|string|min:4',
       ]);

       $sStudentMobileNo = $validatedData['sStudentMobileNo'];
       $sOTP = $validatedData['sOTP'];
       $newPassword = $validatedData['sPassword'];

       $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

       if (!$user || $sOTP !== $user->sOTP) {
           return response()->json(['result' => "8", 'message' => "Invalid OTP or user not found."]);
       }

       $hashedPassword = Hash::make($newPassword);
       DB::table('tblregistration')
           ->where('sStudentMobileNo', $sStudentMobileNo)
           ->update(['sPassword' => $hashedPassword, 'sOTP' => null]);

       return response()->json(['result' => "9", 'message' => "Password reset successful."]);
   }



    
    

}







// sh256

  // public function store(Request $request)
    // {
    //     // Validate the incoming request data
    //     try {
    //         $validatedData = $request->validate([
    //             'sPassword' => 'required|string|max:255',
    //             'sFirstName' => 'required|string|max:255',
    //             'sMiddleName' => 'nullable|string|max:255',
    //             'sLastName' => 'required|string|max:255',
    //             'sGender' => 'required|string|in:Male,Female,other',
    //             'sCity' => 'nullable|string|max:255',
    //             'sStudentMobileNo' => 'required|string|max:15|unique:tblregistration,sStudentMobileNo',
    //             'sParentMobileNo' => 'nullable|string|max:15',
    //             'sEmailId' => 'nullable|email|max:255|unique:tblregistration,sEmailId',
    //             'iAmount' => 'required|integer|min:0',
    //             'iPartialAmount' => 'nullable|integer|min:0',
    //             'iConcession' => 'nullable|integer|min:0',
    //             'iTotalAmount' => 'required|integer|min:0',
    //             'iCounsellingPackageId' => 'nullable|array',
    //             'iCounsellingPackageId.*' => 'integer',
    //             'iAppPackageId' => 'nullable|array',
    //             'iAppPackageId.*' => 'integer',
    //             'sActiveStatus' => 'required|string|in:active,inactive',
    //             'iBisActive' => 'required|boolean',
    //             'sOtherDetails1' => 'nullable|string|max:255',
    //             'sOtherDetails2' => 'nullable|string|max:255',
    //             'sOtherDetails3' => 'nullable|string|max:255',
    //             'sOtherDetails' => 'nullable|string|max:255',
    //         ]);
    //     } catch (ValidationException $e) {
    //         return response()->json([
    //             'result' => 5, 
    //             'message' => 'Validation failed', 
    //             'errors' => $e->errors()
    //         ], 422);
    //     }
    
    //     // Generate a secure password
    //     try {
    //         $salt = bin2hex(random_bytes(16));
    //         $encryptedPassword = hash('sha256', $validatedData['sPassword'] . $salt);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'result' => 0, 
    //             'message' => 'Error generating password', 
    //             'error' => $e->getMessage()
    //         ], 500);
    //     }
    
    //     // Registration data
    //     $registrationData = [
    //         'sPassword' => $encryptedPassword,
    //         'Salt' => $salt,
    //         'sFirstName' => $validatedData['sFirstName'],
    //         'sMiddleName' => $validatedData['sMiddleName'] ?? null,
    //         'sLastName' => $validatedData['sLastName'],
    //         'sGender' => $validatedData['sGender'],
    //         'sCity' => $validatedData['sCity'] ?? null,
    //         'sStudentMobileNo' => $validatedData['sStudentMobileNo'],
    //         'sParentMobileNo' => $validatedData['sParentMobileNo'] ?? null,
    //         'sEmailId' => $validatedData['sEmailId'] ?? null,
    //         'iAmount' => $validatedData['iAmount'],
    //         'iPartialAmount' => $validatedData['iPartialAmount'] ?? null,
    //         'iConcession' => $validatedData['iConcession'] ?? null,
    //         'iTotalAmount' => $validatedData['iTotalAmount'],
    //         'iCounsellingPackageId' => json_encode($validatedData['iCounsellingPackageId'] ?? []),
    //         'iAppPackageId' => json_encode($validatedData['iAppPackageId'] ?? []),
    //         'sActiveStatus' => $validatedData['sActiveStatus'],
    //         'iBisActive' => $validatedData['iBisActive'],
    //         'dtUpdatedDate' => now(),
    //         'sOtherDetails1' => $validatedData['sOtherDetails1'] ?? null,
    //         'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
    //         'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
    //         'sOtherDetails' => $validatedData['sOtherDetails'] ?? null,
    //     ];
    
    //     try {
    //         DB::beginTransaction();
    
    //         // Insert into tblregistration
    //         $iId = DB::table('tblregistration')->insertGetId($registrationData);
    
    //         // Insert into tblAppUser using the registered user's ID
    //         $profileData = [
    //             'iStudentid' => $iId,  // Use the generated ID from tblregistration
    //             'sName' => $validatedData['sFirstName'],  // Map sFirstName to sName    
    //             'dtUpdatedDate' => now(),
    //         ];
    //         DB::table('tblAppUser')->insert($profileData);
    
    //         DB::commit();
    
    //         return response()->json([
    //             'result' => 8,
    //             'message' => 'Registration and profile created successfully',
    //             'id' => $iId,
    //             'password' => $validatedData['sPassword'], // Include the original password
    //             'registration_data' => $registrationData,
    //             'profile_data' => $profileData
    //         ], 201);
    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         return response()->json([
    //             'result' => 0,
    //             'message' => 'Error occurred while creating registration',
    //             'error' => $e->getMessage()
    //         ], 500);
    //     }
    
    
    // }

// login api 
    // public function login(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sStudentMobileNo' => 'required|string|max:15',
    //         'sPassword' => 'required|string|min:4',
    //         'iDeviceId' => 'required|string', // Ensure that the device ID is provided
    //     ]);
    
    //     $sStudentMobileNo = $validatedData['sStudentMobileNo'];
    //     $sPassword = $validatedData['sPassword'];
    //     $iDeviceId = $validatedData['iDeviceId'];
    
    //     $output = [];
    
    //     // Fetch user from tblregistration and ensure the account is active
    //     $user = DB::table('tblregistration')
    //         ->where('sStudentMobileNo', $sStudentMobileNo)
    //         ->where('iBisActive', 1)
    //         ->first();
    
    //     if ($user) {
    //         // Validate password
    //         $hash = hash('sha256', $sPassword . $user->Salt);
    
    //         // Check if the password is correct
    //         $userWithValidPassword = DB::table('tblregistration')
    //             ->select('sStudentMobileNo', 'iId', 'sFirstName', 'iAmount', 'iCounsellingPackageId', 'iAppPackageId')
    //             ->where('sStudentMobileNo', $sStudentMobileNo)
    //             ->where('sPassword', $hash)
    //             ->first();
    
    //         if ($userWithValidPassword) {
    //             // Fetch subscription details
    //             $subscription = DB::table('tblSubscription')
    //                 ->where('iStudentId', $userWithValidPassword->iId)
    //                 ->first();
    
    //             if (!$subscription) {
    //                 // If no subscription exists, just return sFlag = 'Pending' without inserting
    //                 $output['subscription'] = [
    //                     'sStatus' => 'pending',
    //                     'sFlag' => 'NoSubscription'
    //                 ];
    //             } else {
    //                 // If a subscription exists, return its actual sFlag value
    //                 $output['subscription'] = [
    //                     'sStatus' => $subscription->sStatus,
    //                     'sFlag' => $subscription->sFlag
    //                 ];
    //             }
    
    //             // Successful login response
    //             $output['result'] = "8";
    //             $output['message'] = "Login successful.";
    //             $output['records'] = $userWithValidPassword;
    //         } else {
    //             $output['result'] = "10"; // Invalid password
    //             $output['message'] = "Invalid password. Please try again.";
    //         }
    //     } else {
    //         $output['result'] = "5"; // User not found
    //         $output['message'] = "Invalid mobile number. User not found.";
    //     }
    
    //     return response()->json($output);
    // }
    

    // // Forget Password API
    // public function forgetPassword(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sStudentMobileNo' => 'required|string|max:15',
    //     ]);

    //     $sStudentMobileNo = $validatedData['sStudentMobileNo'];

    //     // Check if user exists in the database
    //     $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

    //     $response = [];

    //     if ($user) {
    //         // Generate a random OTP
    //         $otp = rand(100000, 999999);
            
    //         // Update sPassword field with the OTP (as you want to store OTP as password)
    //         DB::table('tblregistration')
    //             ->where('sStudentMobileNo', $sStudentMobileNo)
    //             ->update([
    //                 'sPassword' => $otp,  // Store OTP in sPassword
    //             ]);

    //         // Send OTP to the user (you could send it via SMS, email, etc.)
    //         // You can add that logic here. For now, just including OTP in response.

    //         $response['result'] = "8";  // OTP generated
    //         $response['otp'] = $otp;
    //     } else {
    //         $response['result'] = "5";  // User not found
    //         $response['message'] = "Mobile number not found.";
    //     }

    //     return response()->json($response);
    // }



    // // Reset Password API
    // public function resetPassword(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sStudentMobileNo' => 'required|string|max:15',
    //         'sOTP' => 'required|string',
    //         'sPassword' => 'required|string|min:4',
    //     ]);

    //     $sStudentMobileNo = $validatedData['sStudentMobileNo'];
    //     $sOTP = $validatedData['sOTP'];  // The OTP provided by the user
    //     $newPassword = $validatedData['sPassword'];  // The new password to set

    //     $response = [];

    //     // Find the user in the database by mobile number
    //     $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

    //     // If user not found, return an error response
    //     if (!$user) {
    //         $response['result'] = "5";
    //         $response['message'] = "Mobile number not found.";
    //         return response()->json($response);
    //     }

    //     // Check if the OTP matches the stored value in sPassword
    //     if ($sOTP !== (string) $user->sPassword) {
    //         $response['result'] = "8";
    //         $response['message'] = "Invalid OTP provided.";
    //         return response()->json($response);
    //     }

    //     try {
    //         // Generate a new salt for the password
    //         $newSalt = bin2hex(random_bytes(16));
    //         $encryptedPassword = hash('sha256', $newPassword . $newSalt);

    //         // Update the password with the new encrypted password
    //         $updateStatus = DB::table('tblregistration')
    //             ->where('sStudentMobileNo', $sStudentMobileNo)
    //             ->update([
    //                 'sPassword' => $encryptedPassword,  // Save the new password
    //                 'Salt' => $newSalt,  // Save the new salt
    //                 // Do not include 'sOTP' here as it's not part of the database
    //             ]);

    //         if ($updateStatus) {
    //             $response['result'] = "9";
    //             $response['message'] = "Password has been reset successfully.";
    //         } else {
    //             $response['result'] = "5";
    //             $response['message'] = "Failed to update the password.";
    //         }
    //     } catch (Exception $e) {
    //         $response['result'] = "5";
    //         $response['message'] = "An error occurred: " . $e->getMessage();
    //     }

    //     return response()->json($response);
    // }

    







// Another method
// // Forget Password API
    // public function forgetPassword(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sMobileNo' => 'required|string|max:15',
    //     ]);
    
    //     $sMobileNo = $validatedData['sMobileNo'];
    
    //     // Check if user exists in the database
    //     $user = DB::table('tblregistration')->where('sMobileNo', $sMobileNo)->where('iBisActive', 1)->first();
    
    //     $response = [];
    
    //     if ($user) {
    //         // Generate a random OTP
    //         $otp = rand(100000, 999999);
            
    //         // Update sPassword field with the OTP (as you want to store OTP as password)
    //         DB::table('tblregistration')
    //             ->where('sMobileNo', $sMobileNo)
    //             ->update([
    //                 'sPassword' => $otp,  // Store OTP in sPassword
    //             ]);
    
    //         // Send OTP to the user (you could send it via SMS, email, etc.)
    //         // You can add that logic here. For now, just including OTP in response.
    
    //         $response['result'] = "8";  // OTP generated
    //         $response['otp'] = $otp;
    //     } else {
    //         $response['result'] = "5";  // User not found
    //         $response['message'] = "Mobile number not found.";
    //     }
    
    //     return response()->json($response);
    // }
    


    // // Reset Password API
    // public function resetPassword(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sMobileNo' => 'required|string|max:15',
    //         'sOTP' => 'required|string',
    //         'sPassword' => 'required|string|min:4',
    //     ]);
    
    //     $sMobileNo = $validatedData['sMobileNo'];
    //     $sOTP = $validatedData['sOTP'];  // The OTP provided by the user
    //     $newPassword = $validatedData['sPassword'];  // The new password to set
    
    //     $response = [];
    
    //     // Find the user in the database by mobile number
    //     $user = DB::table('tblregistration')->where('sMobileNo', $sMobileNo)->where('iBisActive', 1)->first();
    
    //     // If user not found, return an error response
    //     if (!$user) {
    //         $response['result'] = "5";
    //         $response['message'] = "Mobile number not found.";
    //         return response()->json($response);
    //     }
    
    //     // Check if the OTP matches the stored value in sPassword
    //     if ($sOTP !== (string) $user->sPassword) {
    //         $response['result'] = "8";
    //         $response['message'] = "Invalid OTP provided.";
    //         return response()->json($response);
    //     }
    
    //     try {
    //         // Generate a new salt for the password
    //         $newSalt = bin2hex(random_bytes(16));
    //         $encryptedPassword = hash('sha256', $newPassword . $newSalt);
    
    //         // Update the password with the new encrypted password
    //         $updateStatus = DB::table('tblregistration')
    //             ->where('sMobileNo', $sMobileNo)
    //             ->update([
    //                 'sPassword' => $encryptedPassword,  // Save the new password
    //                 'Salt' => $newSalt,  // Save the new salt
    //                 // Do not include 'sOTP' here as it's not part of the database
    //             ]);
    
    //         if ($updateStatus) {
    //             $response['result'] = "9";
    //             $response['message'] = "Password has been reset successfully.";
    //         } else {
    //             $response['result'] = "5";
    //             $response['message'] = "Failed to update the password.";
    //         }
    //     } catch (Exception $e) {
    //         $response['result'] = "5";
    //         $response['message'] = "An error occurred: " . $e->getMessage();
    //     }
    
    //     return response()->json($response);
    // }
    



    // // Update the images
    // public function updateRegistration(Request $request)
    // {
    //     // Validate the incoming data
    //     $validatedData = $request->validate([
    //         'updateData.studentId' => 'required|exists:tblregistration,iId',
    //         'updateData.fieldName' => 'required|string|in:sPhoto,sCastCertificate,sAdhaarCard',  // Validate column names
    //         'updateData.imageUrl' => 'required|string',
    //     ]);
        
    //     // Access the validated data correctly
    //     $studentId = $validatedData['updateData']['studentId'];
    //     $fieldName = $validatedData['updateData']['fieldName'];
    //     $imageUrl = $validatedData['updateData']['imageUrl'];
    
    //     // Find the student by iId (primary key)
    //     $student = DB::table('tblregistration')->where('iId', $studentId)->first();
        
    //     if ($student) {
    //         // Dynamically set the field name (e.g., sPhoto, sCastCertificate)
    //         // Validate that the column name exists in the table schema
    //         if (Schema::hasColumn('tblregistration', $fieldName)) {
    //             // Update the student's field with the new image URL
    //             $student->{$fieldName} = $imageUrl;
                
    //             // Log the update process
    //             Log::info('Student record updated', [
    //                 'student_id' => $student->iId, 
    //                 'updated_field' => $fieldName,
    //                 'new_value' => $imageUrl,
    //             ]);
                
    //             // Save the updated record in the database
    //             DB::table('tblregistration')
    //                 ->where('iId', $studentId)
    //                 ->update([$fieldName => $imageUrl]);
    
    //             return response()->json([
    //                 'success' => true,
    //                 'message' => 'Registration table updated successfully'
    //             ]);
    //         } else {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Invalid field name'
    //             ], 400);
    //         }
    //     }
    
    //     return response()->json([
    //         'success' => false,
    //         'message' => 'Student not found'
    //     ], 404);
    // }
    
    // public function login(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sStudentMobileNo' => 'required|string|max:15',
    //         'sPassword' => 'required|string|min:4',
    //         'iDeviceId' => 'required|string', // Ensure that the device ID is provided
    //     ]);

    //     $sStudentMobileNo = $validatedData['sStudentMobileNo'];
    //     $sPassword = $validatedData['sPassword'];
    //     $iDeviceId = $validatedData['iDeviceId'];

    //     $output = [];

    //     // Fetch user from tblregistration based on mobile number and check if the account is active
    //     $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

    //     if ($user) {
    //         // Validate password
    //         $hash = hash('sha256', $sPassword . $user->Salt);

    //         // Check if password is correct
    //         $userWithValidPassword = DB::table('tblregistration')
    //             ->select('sStudentMobileNo', 'iId', 'sFirstName','iAmount','iCounsellingPackageId','iAppPackageId')
    //             ->where('sStudentMobileNo', $sStudentMobileNo)
    //             ->where('sPassword', $hash)
    //             ->first();

    //         if ($userWithValidPassword) {
    //             // Fetch user profile data from tblAppUser
    //             $userProfile = DB::table('tblAppUser')
    //                 ->where('iStudentId', $userWithValidPassword->iId) // Assuming iStudentId is common between tblregistration and tblAppUser
    //                 ->first();

    //             if ($userProfile) {
    //                 // If it's the user's first login, save the iDeviceId in the tblAppUser table
    //                 if (is_null($userProfile->iDeviceId)) {
    //                     // Update the device ID for the first login
    //                     DB::table('tblAppUser')
    //                         ->where('iStudentId', $userWithValidPassword->iId)
    //                         ->update(['iDeviceId' => $iDeviceId]);

    //                     // Return successful login response
    //                     $output['result'] = "8"; // Successful login
    //                     $output['message'] = "First-time login successful. Device ID has been saved.";
    //                     $output['records'] = $userWithValidPassword;
    //                 } else {
    //                     // If the device ID doesn't match, deny login
    //                     if ($userProfile->iDeviceId !== $iDeviceId) {
    //                         $output['result'] = "9"; // Device mismatch error
    //                         $output['message'] = "You are not allowed to log in from a different device.";
    //                     } else {
    //                         // Device ID matches, allow login
    //                         $output['result'] = "8"; // Successful login
    //                         $output['records'] = $userWithValidPassword;
    //                     }
    //                 }
    //             } else {
    //                 $output['result'] = "7"; // User profile not found
    //                 $output['message'] = "User profile not found in the database.";
    //             }
    //         } else {
    //             $output['result'] = "10"; // Invalid password
    //             $output['message'] = "Invalid password. Please try again.";
    //         }
    //     } else {
    //         $output['result'] = "5"; // User not found
    //         $output['message'] = "Invalid mobile number. User not found.";
    //     }

    //     return response()->json($output);
    // }