<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;


class SeatMatrixController extends Controller
{

    // Add Seat Matrix
    public function addSeatMatrix(Request $request)
    {
        // Validate the incoming request data
        $request->validate([
            'sProgram' => 'required|string',
            'sState' => 'required|string',
            'sQuota' => 'required|string',
            'sCourseName' => 'required|string',
            'sRound' => 'required|string',
            'sYear' => 'required|array', // Ensures sYear is an array
            'sYear.*.title' => 'required|string', // Validate each year's title
            'sYear.*.url' => 'required|url', // Validate each year's sampleTemplateUrl
        ]);
    
        // Extract required fields from the request
        $data = $request->only([
            'sProgram', 'sState', 'sQuota', 'sCourseName', 'sRound'
        ]);
    
        // Set default values
        $data['dtUpdatedDate'] = now();
        $data['iBisActive'] = 1;
    
        // Process 'sYear' and encode as JSON
        $data['sYear'] = json_encode($request->input('sYear'));
    
        // Insert the data into the database and get the inserted ID
        try {
            $id = DB::table('tblSeatMatrix')->insertGetId($data);
            return response()->json(['result' => '2', 'message' => 'Seat Matrix Added', 'id' => $id], 201);
        } catch (\Exception $e) {
            // Handle errors gracefully
            return response()->json(['result' => '0', 'message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    



    // Add Muiltiple Seat Matrix
    public function addMultipleSeatMatrices(Request $request)
    {
        try {
            // Validate the incoming request data
            $request->validate([
                'seatMatrices' => 'required|array', // Ensure it's an array of seat matrix entries
                'seatMatrices.*.sProgram' => 'required|string',
                'seatMatrices.*.sState' => 'required|string',
                'seatMatrices.*.sQuota' => 'required|string',
                'seatMatrices.*.sCourseName' => 'required|string',
                'seatMatrices.*.sRound' => 'required|string',
                'seatMatrices.*.sYear' => 'required|array', // Validate the years array
                'seatMatrices.*.sYear.*.title' => 'required|string', // Each year title
                'seatMatrices.*.sYear.*.url' => 'required|url', // Each year URL
            ]);
    
            // Extract and process each seat matrix entry
            $seatMatrices = $request->input('seatMatrices');
            $status = $request->input('status', 'append'); // Default to 'append'
            $sProgram = $request->input('sProgram', []); // Program filter for conditional delete
    
            \Log::info('Received status:', ['status' => $status]);
            \Log::info('Received sProgram:', ['sProgram' => $sProgram]);
    
            // Handle "addall" status - delete records based on program or clear the table
            if ($status === 'addall') {
                if (!empty($sProgram) && is_array($sProgram)) {
                    \Log::info('Programs to delete:', ['programs' => $sProgram]);
    
                    // Delete records for specific programs
                    foreach ($sProgram as $program) {
                        $deletedRows = DB::table('tblSeatMatrix')->where('sProgram', $program)->delete();
                        \Log::info('Deleted rows for program', ['program' => $program, 'deleted' => $deletedRows]);
                    }
                } else {
                    // If no specific program provided, truncate the entire table
                    \Log::info('No programs provided, truncating the entire table.');
                    DB::table('tblSeatMatrix')->truncate();
                }
            }
    
            // Process seat matrices for insertion
            $currentDate = now(); // Get the current timestamp for dtUpdatedDate
            $insertData = []; // Array to hold batch insert data
    
            foreach ($seatMatrices as $matrix) {
                $insertData[] = [
                    'sProgram' => $matrix['sProgram'],
                    'sState' => $matrix['sState'],
                    'sQuota' => $matrix['sQuota'],
                    'sCourseName' => $matrix['sCourseName'],
                    'sRound' => $matrix['sRound'],
                    'sYear' => json_encode($matrix['sYear']), // Encode the sYear field as JSON
                    'dtUpdatedDate' => $currentDate,
                    'iBisActive' => 1,
                ];
            }
    
            // Insert all data into the database in chunks to handle large datasets
            $batchSize = 1800;
            $chunks = array_chunk($insertData, $batchSize);
    
            foreach ($chunks as $chunk) {
                DB::table('tblSeatMatrix')->insert($chunk);
            }
    
            \Log::info('Multiple Seat Matrices Added successfully.');
    
            return response()->json(['result' => '2', 'message' => 'Multiple Seat Matrices Added'], 201);
        } catch (\Exception $e) {
            // Log the error for debugging purposes
            \Log::error('Error in addMultipleSeatMatrices:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            // Return a proper error response
            return response()->json(['result' => '0', 'message' => 'An error occurred while adding seat matrices.'], 500);
        }
    }
    
    



      // Delete a record
      public function deleteSeatMatrix($iId)
      {
          $deleted = DB::table('tblSeatMatrix')->where('iId', $iId)->update(['iBisActive' => 0]);
          if ($deleted) {
              return response()->json(['result'=>'4','message' => 'Seat Matrix deleted']);
          } else {
              return response()->json(['result'=>'5' ,'message' => 'Seat Matrix not found'], 404);
          }
      }



    // get Matrices according To Filter
    public function getSeatMatrices(Request $request)
    {
        // Extract filters from "filters" key in request
        $filters = $request->input('filters', []);
    
        // Ensure we got an array of filters
        if (!is_array($filters) || empty($filters)) {
            return response()->json(['result' => '0', 'message' => 'Filters must be provided in the correct format.'], 400);
        }
    
        // Remove empty filters (trim spaces before checking)
        $filters = array_filter($filters, function ($value) {
            return !empty(trim($value)); // Trim whitespace and check for empty values
        });
    
        // Ensure at least one filter is provided
        if (empty($filters)) {
            return response()->json(['result' => '0', 'message' => 'At least one valid filter must be provided.'], 400);
        }
    
        try {
            $query = DB::table('tblSeatMatrix');
    
            // Apply only non-empty filters
            foreach ($filters as $key => $value) {
                $query->where($key, $value);
            }
    
            $results = $query->get();
    
            // Decode JSON fields if needed
            $decodedResults = $results->map(function ($item) {
                $item->sYear = json_decode($item->sYear);
                return $item;
            });
    
            return response()->json(['result' => '2', 'data' => $decodedResults], 200);
    
        } catch (\Exception $e) {
            return response()->json(['result' => '0', 'message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    
    

    // get Distinct Quota
    public function getDistinctQuota(Request $request)
    {
        // Retrieve 'sProgram' and 'sState' from the request
        $sProgram = $request->input('sProgram');
        $sState = $request->input('sState');
        
        // Build the query
        $query = DB::table('tblSeatMatrix')
            ->select(DB::raw('DISTINCT sQuota'), DB::raw('COUNT(sQuota) as count'))
            ->where('iBisActive', 1)
            ->whereNotNull('sQuota');
        
        // Apply additional filters if 'sProgram' and 'sState' are provided
        if ($sProgram) {
            $query->where('sProgram', $sProgram);  // Filter by sProgram
        }
        if ($sState) {
            $query->where('sState', $sState);  // Filter by sState
        }
        
        // Group by 'sQuota' to get the distinct values and count
        $results = $query->groupBy('sQuota')->get();
        
        // Return the results as JSON
        return response()->json($results);
    }




    // get Distinct state
    public function getDistinctState(Request $request)
    {
        // Get the sProgram from the request
        $sProgram = $request->input('sProgram'); // Assuming the request contains 'sProgram'

        // Perform the query with sProgram filter and retrieve distinct sState
        $results = DB::table('tblSeatMatrix')
            ->select(DB::raw('DISTINCT sState'), DB::raw('COUNT(sState) as count'))
            ->where('iBisActive', 1)
            ->whereNotNull('sState')
            ->where('sProgram', $sProgram) // Filter by sProgram
            ->groupBy('sState')
            ->get();
        
        // Return the results as JSON
        return response()->json($results);
    }


    // get courses
    public function getDistinctCourseNames(Request $request)
    {
        // Retrieve input parameters from the request
        $sProgram = $request->input('sProgram');
        $sState = $request->input('sState');
        $sQuota = $request->input('sQuota');
        
        // Start building the query
        $query = DB::table('tblSeatMatrix')
            ->select(DB::raw('DISTINCT sCourseName')) // Fetch distinct course names
            ->where('iBisActive', 1)
            ->whereNotNull('sCourseName'); // Ensure course names are not null
        
        // Apply filters for the provided parameters
        if ($sProgram) {
            $query->where('sProgram', $sProgram); // Filter by sProgram
        }
        if ($sState) {
            $query->where('sState', $sState); // Filter by sState
        }
        if ($sQuota) {
            $query->where('sQuota', $sQuota); // Filter by sQuota
        }
        
        // Get the distinct course names
        $results = $query->get();
        
        // Return the results as a JSON response
        return response()->json($results);
    }




    // get distinct Round
    public function getDistinctRound()
    {
        $results = DB::table('tblSeatMatrix')
            ->select(DB::raw('DISTINCT sRound'), DB::raw('COUNT(sRound) as count'))
            ->where('iBisActive', 1)
            ->whereNotNull('sRound') 
            ->groupBy('sRound')
            ->get();
        
        return response()->json($results);
    }



    // Get all records
    public function getSeatMatricesAll()
    {
        $seatMatrices = DB::table('tblSeatMatrix')->where('iBisActive', 1)->get();
    
        // Decode sYear for each record
        foreach ($seatMatrices as $seatMatrix) {
            if (!empty($seatMatrix->sYear)) {
                $decodedYear = json_decode($seatMatrix->sYear, true);
                $seatMatrix->sYear = json_last_error() === JSON_ERROR_NONE ? $decodedYear : [];
            }
        }
    
        return response()->json(['result' => '1', 'data' => $seatMatrices]);
    }
    


    // Update Seat Matrix
    public function updateSeatMatrix(Request $request)
    {
        // Decode sYear if it's a JSON string
        if ($request->has('sYear') && is_string($request->input('sYear'))) {
            $decodedYear = json_decode($request->input('sYear'), true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $request->merge(['sYear' => $decodedYear]);
            }
        }

        // Validate the incoming request data
        $request->validate([
            'iId' => 'required|integer', // Ensure iId is provided for updating
            'sProgram' => 'sometimes|required|string',
            'sState' => 'sometimes|required|string',
            'sQuota' => 'sometimes|required|string',
            'sCourseName' => 'sometimes|required|string',
            'sRound' => 'sometimes|required|string',
            'sYear' => 'sometimes|required|array', // Ensures sYear is an array if provided
            'sYear.*.title' => 'sometimes|required|string', // Validate each year's title
            'sYear.*.url' => 'sometimes|required|url', // Validate each year's sampleTemplateUrl
        ]);

        // Extract iId from request
        $iId = $request->input('iId');
        
        // Extract only the fields that are present in the request
        $data = $request->only(['sProgram', 'sState', 'sQuota', 'sCourseName', 'sRound']);
        
        // Set updated date
        $data['dtUpdatedDate'] = now();

        // Process 'sYear' if provided and encode as JSON
        if ($request->has('sYear')) {
            $data['sYear'] = json_encode($request->input('sYear'));
        }

        try {
            // Perform the update
            $updated = DB::table('tblSeatMatrix')->where('iId', $iId)->update($data);
            
            if ($updated) {
                return response()->json(['result' => '3', 'message' => 'Seat Matrix Updated'], 200);
            } else {
                return response()->json(['result' => '5', 'message' => 'No changes made or invalid ID'], 404);
            }
        } catch (\Exception $e) {
            // Handle errors gracefully
            return response()->json(['result' => '5', 'message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }



}
