<?php

namespace App\Http\Controllers\App;

use Illuminate\Support\Facades\Schema;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;


class appUserController extends Controller
{

    
    // Get all records from tblregistration
    public function index()
    {
        $registrations = DB::table('tblregistration')->where('iBisActive', 1)->get();
    
        if ($registrations->isEmpty()) {
            return response()->json([
                'result' => "6",
                'message' => 'No active registrations found',
            ], 404);  
        }

    
        // Return the results as JSON with success status
        return response()->json([
            'result' => "1",
            'message' => 'Active registrations retrieved successfully',
            'data' => $registrations,
        ], 200);  
    }
    

    // Get a specific record by ID
    public function show($id)
    {
        // Fetch data from the database
        $registration = DB::table('tblregistration as r')
            ->leftJoin('tblAppUser as u', 'r.iId', '=', 'u.iStudentId')
            ->select(
                'r.*', // All registration columns
                'u.sName', 'u.sAddress', 'u.iDeviceId', 'u.sPCMmarks', 'u.sPCBmarks', 
                'u.fSSCMarks', 'u.fHSCMarks', 'u.fNeetMarks', 'u.sJEEMarks', 'u.sCast' // All AppUser columns
            )
            ->where('r.iId', $id)
            ->where('r.iBisActive', 1)
            ->first();
    
        // Ensure data exists
        if ($registration) {
            // Check and properly decode JSON fields
            foreach (['iCounsellingPackageId', 'iAppPackageId'] as $field) {
                if (!empty($registration->$field) && is_string($registration->$field)) {
                    $decoded = ($registration->$field);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $registration->$field = $decoded; // Convert string to array
                    } else {
                        Log::error("JSON Decode Error for {$field}", ['value' => $registration->$field]);
                    }
                }
            }
    
            // Return correctly formatted JSON
            return response()->json([
                'result' => "1",
                'data' => $registration
            ], 200, [], JSON_UNESCAPED_UNICODE);
        } 
        
        // Return error response if data not found
        return response()->json([
            'result' => "6",
            'message' => 'Registration not found'
        ], 404);
    }
    
    


    // Add a new registration
    public function store(Request $request)
    {
        // Validate the incoming request data
        try {
            $validatedData = $request->validate([
                'sPassword' => 'required|string|max:255',
                'sFirstName' => 'required|string|max:255',
                'sMiddleName' => 'nullable|string|max:255',
                'sLastName' => 'required|string|max:255',
                'sGender' => 'required|string|in:male,female,other',
                'sCity' => 'nullable|string|max:255',
                'sStudentMobileNo' => 'required|string|max:15|unique:tblregistration,sStudentMobileNo',
                'sParentMobileNo' => 'nullable|string|max:15',
                'sEmailId' => 'nullable|email|max:255|unique:tblregistration,sEmailId',
                'iAmount' => 'required|integer|min:0',
                'iPartialAmount' => 'nullable|integer|min:0',
                'iConcession' => 'nullable|integer|min:0',
                'iTotalAmount' => 'required|integer|min:0',
                'iCounsellingPackageId' => 'nullable|array',
                'iCounsellingPackageId.*' => 'integer',
                'iAppPackageId' => 'nullable|array',
                'iAppPackageId.*' => 'integer',
                'sActiveStatus' => 'required|string|in:active,inactive',
                'iBisActive' => 'required|boolean',
                'sOtherDetails1' => 'nullable|string|max:255',
                'sOtherDetails2' => 'nullable|string|max:255',
                'sOtherDetails3' => 'nullable|string|max:255',
                'sOtherDetails' => 'nullable|string|max:255',
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'result' => 5, 
                'message' => 'Validation failed', 
                'errors' => $e->errors()
            ], 422);
        }
    
        // Generate a secure password
        try {
            $salt = bin2hex(random_bytes(16));
            $encryptedPassword = hash('sha256', $validatedData['sPassword'] . $salt);
        } catch (\Exception $e) {
            return response()->json([
                'result' => 0, 
                'message' => 'Error generating password', 
                'error' => $e->getMessage()
            ], 500);
        }
    
        // Registration data
        $registrationData = [
            'sPassword' => $encryptedPassword,
            'Salt' => $salt,
            'sFirstName' => $validatedData['sFirstName'],
            'sMiddleName' => $validatedData['sMiddleName'] ?? null,
            'sLastName' => $validatedData['sLastName'],
            'sGender' => $validatedData['sGender'],
            'sCity' => $validatedData['sCity'] ?? null,
            'sStudentMobileNo' => $validatedData['sStudentMobileNo'],
            'sParentMobileNo' => $validatedData['sParentMobileNo'] ?? null,
            'sEmailId' => $validatedData['sEmailId'] ?? null,
            'iAmount' => $validatedData['iAmount'],
            'iPartialAmount' => $validatedData['iPartialAmount'] ?? null,
            'iConcession' => $validatedData['iConcession'] ?? null,
            'iTotalAmount' => $validatedData['iTotalAmount'],
            'iCounsellingPackageId' => json_encode($validatedData['iCounsellingPackageId'] ?? []),
            'iAppPackageId' => json_encode($validatedData['iAppPackageId'] ?? []),
            'sActiveStatus' => $validatedData['sActiveStatus'],
            'iBisActive' => $validatedData['iBisActive'],
            'dtUpdatedDate' => now(),
            'sOtherDetails1' => $validatedData['sOtherDetails1'] ?? null,
            'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
            'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
            'sOtherDetails' => $validatedData['sOtherDetails'] ?? null,
        ];
    
        try {
            DB::beginTransaction();
    
            // Insert into tblregistration
            $iId = DB::table('tblregistration')->insertGetId($registrationData);
    
            // Insert into tblAppUser using the registered user's ID
            $profileData = [
                'iStudentid' => $iId,  // Use the generated ID from tblregistration
                'sName' => $validatedData['sFirstName'],  // Map sFirstName to sName    
                'dtUpdatedDate' => now(),
            ];
            DB::table('tblAppUser')->insert($profileData);
    
            DB::commit();
    
            return response()->json([
                'result' => 8,
                'message' => 'Registration and profile created successfully',
                'id' => $iId,
                'password' => $validatedData['sPassword'], // Include the original password
                'registration_data' => $registrationData,
                'profile_data' => $profileData
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'result' => 0,
                'message' => 'Error occurred while creating registration',
                'error' => $e->getMessage()
            ], 500);
        }
    
    
    }
    
    


    // Update a registration by iId
    public function update(Request $request)
    {
        // Ensure iId is present in the request
        $iId = $request->input('iId');
        if (!$iId) {
            return response()->json(['result' => "1", 'message' => 'iId is required'], 400);
        }
    
        // Validate the incoming request data
        $validatedData = $request->validate([
            'sName' => 'string|max:255|nullable',
            'sMobileNo' => 'string|max:15|nullable',
            'iProgramId' => 'integer|nullable',
            'sPaymentStatus' => 'string|max:255|nullable',
            'sCounsellingPackage' => 'array|nullable',
            'sCounsellingPackage.*' => 'string|max:255|nullable',
            'sAppPackage' => 'array|nullable',
            'sAppPackage.*' => 'string|max:255|nullable',
            'sProgram' => 'array|nullable',
            'sProgram.*' => 'string|max:255|nullable',
            'sCourseName' => 'array|nullable',
            'sCourseName.*' => 'string|max:255|nullable',
            'iBisActive' => 'integer|nullable',
            'sOtherDetails1' => 'string|nullable',
            'sOtherDetails2' => 'string|nullable',
            'sOtherDetails3' => 'string|nullable',
            'sOtherDetails4' => 'string|nullable',
        ]);
    
        // Filter out null values
        $updateData = array_filter($validatedData, function ($value) {
            return !is_null($value);
        });
    
        // Convert array fields to JSON
        $arrayFields = ['sCounsellingPackage', 'sAppPackage', 'sProgram', 'sCourseName'];
        foreach ($arrayFields as $field) {
            if (isset($updateData[$field])) {
                $updateData[$field] = json_encode($updateData[$field]);
            }
        }
    
        // Ensure dtUpdatedDate is updated to current time
        $updateData['dtUpdatedDate'] = now();
    
        // Update the registration in the database
        try {
            $updated = DB::table('tblregistration')->where('iId', $iId)->update($updateData);
    
            if ($updated) {
                return response()->json(['result' => "3", 'message' => 'Registration updated successfully']);
            } else {
                return response()->json(['result' => "5", 'message' => 'No changes made or record not found'], 404);
            }
        } catch (\Exception $e) {
            return response()->json(['result' => "0", 'message' => 'Error updating registration', 'error' => $e->getMessage()], 500);
        }
    }
    
    
    

    // Delete a registration by ID
    public function destroy($iId)
   {
            // Find the registration by ID
            $registration = DB::table('tblregistration')->where('iId', $iId)->first();

            if (!$registration) {
                return response()->json(['result' => 0, 'message' => 'Registration not found'], 404);
            }

            $updated = DB::table('tblregistration')->where('iId', $iId)->update(['iBisActive' => 0]);

            if ($updated) {
                return response()->json(['result' => "4", 'message' => 'Registration marked as inactive successfully'], 200);
            } else {
                return response()->json(['result' => "5", 'message' => 'Failed to mark registration as inactive'], 500);
            }
   }


    
    // login api
    public function login(Request $request)
    {
        $validatedData = $request->validate([
            'sStudentMobileNo' => 'required|string|max:15',
            'sPassword' => 'required|string|min:4',
            'iDeviceId' => 'required|string', // Ensure that the device ID is provided
        ]);
    
        $sStudentMobileNo = $validatedData['sStudentMobileNo'];
        $sPassword = $validatedData['sPassword'];
        $iDeviceId = $validatedData['iDeviceId'];
    
        $output = [];
    
        // Fetch user from tblregistration and ensure the account is active
        $user = DB::table('tblregistration')
            ->where('sStudentMobileNo', $sStudentMobileNo)
            ->where('iBisActive', 1)
            ->first();
    
        if ($user) {
            // Validate password
            $hash = hash('sha256', $sPassword . $user->Salt);
    
            // Check if the password is correct
            $userWithValidPassword = DB::table('tblregistration')
                ->select('sStudentMobileNo', 'iId', 'sFirstName', 'iAmount', 'iCounsellingPackageId', 'iAppPackageId')
                ->where('sStudentMobileNo', $sStudentMobileNo)
                ->where('sPassword', $hash)
                ->first();
    
            if ($userWithValidPassword) {
                // Fetch subscription details
                $subscription = DB::table('tblSubscription')
                    ->where('iStudentId', $userWithValidPassword->iId)
                    ->first();
    
                if (!$subscription) {
                    // If no subscription exists, just return sFlag = 'Pending' without inserting
                    $output['subscription'] = [
                        'sStatus' => 'pending',
                        'sFlag' => 'NoSubscription'
                    ];
                } else {
                    // If a subscription exists, return its actual sFlag value
                    $output['subscription'] = [
                        'sStatus' => $subscription->sStatus,
                        'sFlag' => $subscription->sFlag
                    ];
                }
    
                // Successful login response
                $output['result'] = "8";
                $output['message'] = "Login successful.";
                $output['records'] = $userWithValidPassword;
            } else {
                $output['result'] = "10"; // Invalid password
                $output['message'] = "Invalid password. Please try again.";
            }
        } else {
            $output['result'] = "5"; // User not found
            $output['message'] = "Invalid mobile number. User not found.";
        }
    
        return response()->json($output);
    }
    

    // Forget Password API
    public function forgetPassword(Request $request)
    {
        $validatedData = $request->validate([
            'sStudentMobileNo' => 'required|string|max:15',
        ]);

        $sStudentMobileNo = $validatedData['sStudentMobileNo'];

        // Check if user exists in the database
        $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

        $response = [];

        if ($user) {
            // Generate a random OTP
            $otp = rand(100000, 999999);
            
            // Update sPassword field with the OTP (as you want to store OTP as password)
            DB::table('tblregistration')
                ->where('sStudentMobileNo', $sStudentMobileNo)
                ->update([
                    'sPassword' => $otp,  // Store OTP in sPassword
                ]);

            // Send OTP to the user (you could send it via SMS, email, etc.)
            // You can add that logic here. For now, just including OTP in response.

            $response['result'] = "8";  // OTP generated
            $response['otp'] = $otp;
        } else {
            $response['result'] = "5";  // User not found
            $response['message'] = "Mobile number not found.";
        }

        return response()->json($response);
    }



    // Reset Password API
    public function resetPassword(Request $request)
    {
        $validatedData = $request->validate([
            'sStudentMobileNo' => 'required|string|max:15',
            'sOTP' => 'required|string',
            'sPassword' => 'required|string|min:4',
        ]);

        $sStudentMobileNo = $validatedData['sStudentMobileNo'];
        $sOTP = $validatedData['sOTP'];  // The OTP provided by the user
        $newPassword = $validatedData['sPassword'];  // The new password to set

        $response = [];

        // Find the user in the database by mobile number
        $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

        // If user not found, return an error response
        if (!$user) {
            $response['result'] = "5";
            $response['message'] = "Mobile number not found.";
            return response()->json($response);
        }

        // Check if the OTP matches the stored value in sPassword
        if ($sOTP !== (string) $user->sPassword) {
            $response['result'] = "8";
            $response['message'] = "Invalid OTP provided.";
            return response()->json($response);
        }

        try {
            // Generate a new salt for the password
            $newSalt = bin2hex(random_bytes(16));
            $encryptedPassword = hash('sha256', $newPassword . $newSalt);

            // Update the password with the new encrypted password
            $updateStatus = DB::table('tblregistration')
                ->where('sStudentMobileNo', $sStudentMobileNo)
                ->update([
                    'sPassword' => $encryptedPassword,  // Save the new password
                    'Salt' => $newSalt,  // Save the new salt
                    // Do not include 'sOTP' here as it's not part of the database
                ]);

            if ($updateStatus) {
                $response['result'] = "9";
                $response['message'] = "Password has been reset successfully.";
            } else {
                $response['result'] = "5";
                $response['message'] = "Failed to update the password.";
            }
        } catch (Exception $e) {
            $response['result'] = "5";
            $response['message'] = "An error occurred: " . $e->getMessage();
        }

        return response()->json($response);
    }

    

}
















// // Forget Password API
    // public function forgetPassword(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sMobileNo' => 'required|string|max:15',
    //     ]);
    
    //     $sMobileNo = $validatedData['sMobileNo'];
    
    //     // Check if user exists in the database
    //     $user = DB::table('tblregistration')->where('sMobileNo', $sMobileNo)->where('iBisActive', 1)->first();
    
    //     $response = [];
    
    //     if ($user) {
    //         // Generate a random OTP
    //         $otp = rand(100000, 999999);
            
    //         // Update sPassword field with the OTP (as you want to store OTP as password)
    //         DB::table('tblregistration')
    //             ->where('sMobileNo', $sMobileNo)
    //             ->update([
    //                 'sPassword' => $otp,  // Store OTP in sPassword
    //             ]);
    
    //         // Send OTP to the user (you could send it via SMS, email, etc.)
    //         // You can add that logic here. For now, just including OTP in response.
    
    //         $response['result'] = "8";  // OTP generated
    //         $response['otp'] = $otp;
    //     } else {
    //         $response['result'] = "5";  // User not found
    //         $response['message'] = "Mobile number not found.";
    //     }
    
    //     return response()->json($response);
    // }
    


    // // Reset Password API
    // public function resetPassword(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sMobileNo' => 'required|string|max:15',
    //         'sOTP' => 'required|string',
    //         'sPassword' => 'required|string|min:4',
    //     ]);
    
    //     $sMobileNo = $validatedData['sMobileNo'];
    //     $sOTP = $validatedData['sOTP'];  // The OTP provided by the user
    //     $newPassword = $validatedData['sPassword'];  // The new password to set
    
    //     $response = [];
    
    //     // Find the user in the database by mobile number
    //     $user = DB::table('tblregistration')->where('sMobileNo', $sMobileNo)->where('iBisActive', 1)->first();
    
    //     // If user not found, return an error response
    //     if (!$user) {
    //         $response['result'] = "5";
    //         $response['message'] = "Mobile number not found.";
    //         return response()->json($response);
    //     }
    
    //     // Check if the OTP matches the stored value in sPassword
    //     if ($sOTP !== (string) $user->sPassword) {
    //         $response['result'] = "8";
    //         $response['message'] = "Invalid OTP provided.";
    //         return response()->json($response);
    //     }
    
    //     try {
    //         // Generate a new salt for the password
    //         $newSalt = bin2hex(random_bytes(16));
    //         $encryptedPassword = hash('sha256', $newPassword . $newSalt);
    
    //         // Update the password with the new encrypted password
    //         $updateStatus = DB::table('tblregistration')
    //             ->where('sMobileNo', $sMobileNo)
    //             ->update([
    //                 'sPassword' => $encryptedPassword,  // Save the new password
    //                 'Salt' => $newSalt,  // Save the new salt
    //                 // Do not include 'sOTP' here as it's not part of the database
    //             ]);
    
    //         if ($updateStatus) {
    //             $response['result'] = "9";
    //             $response['message'] = "Password has been reset successfully.";
    //         } else {
    //             $response['result'] = "5";
    //             $response['message'] = "Failed to update the password.";
    //         }
    //     } catch (Exception $e) {
    //         $response['result'] = "5";
    //         $response['message'] = "An error occurred: " . $e->getMessage();
    //     }
    
    //     return response()->json($response);
    // }
    



    // // Update the images
    // public function updateRegistration(Request $request)
    // {
    //     // Validate the incoming data
    //     $validatedData = $request->validate([
    //         'updateData.studentId' => 'required|exists:tblregistration,iId',
    //         'updateData.fieldName' => 'required|string|in:sPhoto,sCastCertificate,sAdhaarCard',  // Validate column names
    //         'updateData.imageUrl' => 'required|string',
    //     ]);
        
    //     // Access the validated data correctly
    //     $studentId = $validatedData['updateData']['studentId'];
    //     $fieldName = $validatedData['updateData']['fieldName'];
    //     $imageUrl = $validatedData['updateData']['imageUrl'];
    
    //     // Find the student by iId (primary key)
    //     $student = DB::table('tblregistration')->where('iId', $studentId)->first();
        
    //     if ($student) {
    //         // Dynamically set the field name (e.g., sPhoto, sCastCertificate)
    //         // Validate that the column name exists in the table schema
    //         if (Schema::hasColumn('tblregistration', $fieldName)) {
    //             // Update the student's field with the new image URL
    //             $student->{$fieldName} = $imageUrl;
                
    //             // Log the update process
    //             Log::info('Student record updated', [
    //                 'student_id' => $student->iId, 
    //                 'updated_field' => $fieldName,
    //                 'new_value' => $imageUrl,
    //             ]);
                
    //             // Save the updated record in the database
    //             DB::table('tblregistration')
    //                 ->where('iId', $studentId)
    //                 ->update([$fieldName => $imageUrl]);
    
    //             return response()->json([
    //                 'success' => true,
    //                 'message' => 'Registration table updated successfully'
    //             ]);
    //         } else {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Invalid field name'
    //             ], 400);
    //         }
    //     }
    
    //     return response()->json([
    //         'success' => false,
    //         'message' => 'Student not found'
    //     ], 404);
    // }
    
    // public function login(Request $request)
    // {
    //     $validatedData = $request->validate([
    //         'sStudentMobileNo' => 'required|string|max:15',
    //         'sPassword' => 'required|string|min:4',
    //         'iDeviceId' => 'required|string', // Ensure that the device ID is provided
    //     ]);

    //     $sStudentMobileNo = $validatedData['sStudentMobileNo'];
    //     $sPassword = $validatedData['sPassword'];
    //     $iDeviceId = $validatedData['iDeviceId'];

    //     $output = [];

    //     // Fetch user from tblregistration based on mobile number and check if the account is active
    //     $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

    //     if ($user) {
    //         // Validate password
    //         $hash = hash('sha256', $sPassword . $user->Salt);

    //         // Check if password is correct
    //         $userWithValidPassword = DB::table('tblregistration')
    //             ->select('sStudentMobileNo', 'iId', 'sFirstName','iAmount','iCounsellingPackageId','iAppPackageId')
    //             ->where('sStudentMobileNo', $sStudentMobileNo)
    //             ->where('sPassword', $hash)
    //             ->first();

    //         if ($userWithValidPassword) {
    //             // Fetch user profile data from tblAppUser
    //             $userProfile = DB::table('tblAppUser')
    //                 ->where('iStudentId', $userWithValidPassword->iId) // Assuming iStudentId is common between tblregistration and tblAppUser
    //                 ->first();

    //             if ($userProfile) {
    //                 // If it's the user's first login, save the iDeviceId in the tblAppUser table
    //                 if (is_null($userProfile->iDeviceId)) {
    //                     // Update the device ID for the first login
    //                     DB::table('tblAppUser')
    //                         ->where('iStudentId', $userWithValidPassword->iId)
    //                         ->update(['iDeviceId' => $iDeviceId]);

    //                     // Return successful login response
    //                     $output['result'] = "8"; // Successful login
    //                     $output['message'] = "First-time login successful. Device ID has been saved.";
    //                     $output['records'] = $userWithValidPassword;
    //                 } else {
    //                     // If the device ID doesn't match, deny login
    //                     if ($userProfile->iDeviceId !== $iDeviceId) {
    //                         $output['result'] = "9"; // Device mismatch error
    //                         $output['message'] = "You are not allowed to log in from a different device.";
    //                     } else {
    //                         // Device ID matches, allow login
    //                         $output['result'] = "8"; // Successful login
    //                         $output['records'] = $userWithValidPassword;
    //                     }
    //                 }
    //             } else {
    //                 $output['result'] = "7"; // User profile not found
    //                 $output['message'] = "User profile not found in the database.";
    //             }
    //         } else {
    //             $output['result'] = "10"; // Invalid password
    //             $output['message'] = "Invalid password. Please try again.";
    //         }
    //     } else {
    //         $output['result'] = "5"; // User not found
    //         $output['message'] = "Invalid mobile number. User not found.";
    //     }

    //     return response()->json($output);
    // }